/**
 * (c) 2014 FAO / UN (project: fi-security-common)
 */
package org.fao.fi.security.common.encryption.pgp;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.NoSuchProviderException;

import javax.annotation.PostConstruct;
import javax.inject.Inject;
import javax.inject.Named;

import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.fao.fi.security.common.encryption.pgp.exceptions.KeyringAccessException;
import org.fao.fi.security.common.encryption.pgp.exceptions.KeyringException;
import org.fao.fi.security.common.encryption.pgp.exceptions.MissingOrBadKeyringException;
import org.fao.fi.security.common.utilities.FileUtils;
import org.fao.fi.security.common.utilities.LoggingClient;
import org.fao.fi.security.common.utilities.pgp.PGPEncryptor;
import org.fao.fi.security.common.utilities.pgp.configuration.PublicKeyringConfiguration;

/**
 * Place your class / interface description here.
 *
 * History:
 *
 * ------------- --------------- -----------------------
 * Date			 Author			 Comment
 * ------------- --------------- -----------------------
 * 6 May 2014   Fiorellato     Creation.
 *
 * @version 1.0
 * @since 6 May 2014
 */
public class PGPEncryptorBean extends LoggingClient {
	@Inject private PGPEncryptor _delegate;
	@Inject private @Named(PGPBeanConstants.PGP_PUBLIC_KEYRING_CONFIG_BEAN_NAME) PublicKeyringConfiguration _publicKeyringConfigurator;
	
	private PGPPublicKey _publicKey;
	
	/**
	 * Class constructor
	 *
	 */
	public PGPEncryptorBean() {
		super();
	}
	
	/**
	 * Class constructor
	 *
	 * @param publicKeyringConfigurator
	 */
	public PGPEncryptorBean(PublicKeyringConfiguration publicKeyringConfigurator) {
		super();
		
		this._delegate = new PGPEncryptor();
		this._publicKeyringConfigurator = publicKeyringConfigurator;
	}
	
	@PostConstruct private void validate() {
		if(this._delegate == null)
			throw new IllegalArgumentException("The delegate PGP encryptor cannot be null");
		
		if(this._publicKeyringConfigurator == null)
			throw new IllegalArgumentException("The public keyring configurator cannot be null");
	}
	
	public byte[] encryptBytes(byte[] toEncrypt) throws IOException, KeyringException, PGPException, NoSuchProviderException {
		return this._delegate.encryptBytes(toEncrypt, this.lazyReadPublicKey(this._publicKeyringConfigurator.getKeyringResourceURL()));
	}
	
	public void encryptStream(InputStream toEncrypt, OutputStream encrypted) throws IOException, KeyringException, PGPException, NoSuchProviderException {
		this._delegate.encryptStream(toEncrypt, encrypted, this.lazyReadPublicKey(this._publicKeyringConfigurator.getKeyringResourceURL()));
	}
	
	private PGPPublicKey lazyReadPublicKey(String keyringResourceURL) throws KeyringException, PGPException {
		return this._publicKey != null ? 
					  this._publicKey : 
					( this._publicKey = this.readPublicKey(keyringResourceURL) );
	}
	
	private PGPPublicKey readPublicKey(String keyringResourceURL) throws KeyringException, PGPException {
		if(keyringResourceURL == null)
			throw new MissingOrBadKeyringException("No keyring resource URL provided");
		
		InputStream keyringStream = null;
		
		try {
			try {
//				keyringStream = FileUtils.streamResource(keyringResourceURL);
				keyringStream = FileUtils.streamAndCopyResource(keyringResourceURL);
			} catch(FileNotFoundException FNFe) {
				throw new MissingOrBadKeyringException("No keyring available as a resource at URL " + keyringResourceURL, FNFe);
			} catch(IOException IOe) {
				throw new MissingOrBadKeyringException("Unable to access keyring as a resource at URL " + keyringResourceURL + ": " + IOe.getMessage(), IOe);
			}
			
			if(keyringStream == null)
				throw new MissingOrBadKeyringException("No keyring available as a resource at URL " + keyringResourceURL);
			
			PGPPublicKey toReturn = null;
			
			try {
				toReturn = this._delegate.readPublicKeyFromKeyringCollection(keyringStream);
			} catch(IOException IOe) {
				throw new MissingOrBadKeyringException("Unable to extract public key from keyring as a resource at URL " + keyringResourceURL + ": " + IOe.getMessage(), IOe);
			} catch(KeyringException kre) {
				throw new KeyringAccessException("Unable to extract public key from keyring as a resource at URL " + keyringResourceURL + ": " + kre.getMessage(), kre);
			} catch(PGPException pe) {
				throw new KeyringAccessException("Unable to extract public key from keyring as a resource at URL " + keyringResourceURL + ": " + pe.getMessage(), pe);
			}
			
			return toReturn;
		} finally {
			if(keyringStream != null) {
				try {
					keyringStream.close();
				} catch(IOException IOe) {
					this._log.warn("Unable to close input stream for keyring resource at {}: {}", keyringResourceURL, IOe.getMessage(), IOe);
				}
			}
		}
	}
}
