package gr.i2s.bluebridge.simul.calc;

import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;

import com.google.common.base.MoreObjects;
import com.google.common.collect.Ordering;
import com.google.common.collect.RangeMap;
import com.google.common.primitives.Doubles;
import com.google.common.primitives.Longs;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;

import gr.i2s.bluebridge.simul.model.Fcr;
import gr.i2s.bluebridge.simul.model.Mortality;
import gr.i2s.bluebridge.simul.model.Scenario;
import gr.i2s.bluebridge.simul.model.Sfr;

public class ScenarioExecutor {
	private static final Log logger = LogFactoryUtil.getLog(ScenarioExecutor.class);

	private static final SimpleDateFormat FULL_DATE_FORMAT = new SimpleDateFormat("YYYYMMdd-Hms-S");

	Ordering<Fcr> fcrByTempWeight = new Ordering<Fcr>() {
		public int compare(Fcr left, Fcr right) {
			int toRet = Longs.compare(left.getTemperature(), right.getTemperature());
			if (toRet == 0) {
				// descending
				toRet = -1 * Doubles.compare(left.getFromWeight(), right.getFromWeight());
			}
			return toRet;
		}
	};

	Ordering<Sfr> sfrByTempWeight = new Ordering<Sfr>() {
		public int compare(Sfr left, Sfr right) {
			int toRet = Longs.compare(left.getTemperature(), right.getTemperature());
			if (toRet == 0) {
				// descending
				toRet = -1 * Doubles.compare(left.getFromWeight(), right.getFromWeight());
			}
			return toRet;
		}
	};

	Ordering<Mortality> mortalityByTempWeight = new Ordering<Mortality>() {
		public int compare(Mortality left, Mortality right) {
			int toRet = Longs.compare(left.getTemperature(), right.getTemperature());
			if (toRet == 0) {
				// descending
				toRet = -1 * Doubles.compare(left.getFromWeight(), right.getFromWeight());
			}
			return toRet;
		}
	};

	class Daily {
		public Calendar date;
		public Integer temperature; // count
		public Double mab; // gr
		public Integer fishcount; // count
		public Integer dead; // count
		public Double deadBM; // gr
		public Double food; // gr
		public Double growth; // gr
		public Double fcr;
		public Double sfr;
		public Double fcrBiol;
		public Double fcrEcon;

		public Daily(Daily daily) {
			this();
			this.date.setTime(new Date(daily.date.getTimeInMillis()));
			this.temperature = new Integer(daily.temperature);
			this.mab = new Double(daily.mab);
			this.fishcount = new Integer(daily.fishcount);
			this.dead = new Integer(daily.dead);
			this.deadBM = new Double(daily.deadBM);
			this.food = new Double(daily.food);
			this.growth = new Double(daily.growth);
			this.fcr = new Double(daily.fcr);
			this.sfr = new Double(daily.sfr);
			this.fcrBiol = new Double(daily.fcrBiol);
			this.fcrEcon = new Double(daily.fcrEcon);
		}

		public Daily() {
			this.date = Calendar.getInstance();
		}

		@Override
		public String toString() {
			return MoreObjects.toStringHelper(this).add("date", FULL_DATE_FORMAT.format(date.getTime()))
					.add("temperature", temperature).add("mab", mab).add("fishcount", fishcount).add("dead", dead)
					.add("deadBM", deadBM).add("food", food).add("growth", growth).add("fcr", fcr).add("sfr", sfr)
					.add("fcrBiol", fcrBiol).add("fcrEcon", fcrEcon).toString();
		}

	}

	final private Scenario mScenario;
	final private Map<Integer, RangeMap<Double, Double>> fcrTable;
	final private Map<Integer, RangeMap<Double, Double>> sfrTable;
	final private Map<Integer, RangeMap<Double, Double>> mortalityTable;
	final private Map<Long, Integer> temperatureTable;
	final private Integer modelTemperature[];
	final private Queue<Daily> dailyResults;

	public ScenarioExecutor(final Scenario scenario) {
		mScenario = scenario;
		fcrTable = new HashMap<Integer, RangeMap<Double, Double>>();
		sfrTable = new HashMap<Integer, RangeMap<Double, Double>>();
		mortalityTable = new HashMap<Integer, RangeMap<Double, Double>>();
		temperatureTable = new HashMap<>();
		modelTemperature = new Integer[24];
		dailyResults = new LinkedList<Daily>();
	}

	public void run() {
		beforeRun();
		doRun();
		afterRun();
	}

	private void beforeRun() {
		// long start = System.currentTimeMillis();
		//
		// // these should be cached in the session
		// try {
		// SimulModelFullView entity = null;
		// entity =
		// SimulModelFullViewLocalServiceUtil.getSimulModelFullView(mScenario.getSimulModelId());
		// modelTemperature[0] = (int) entity.getPeriodJanA();
		// modelTemperature[1] = (int) entity.getPeriodJanB();
		// modelTemperature[2] = (int) entity.getPeriodFebA();
		// modelTemperature[3] = (int) entity.getPeriodFebB();
		// modelTemperature[4] = (int) entity.getPeriodMarA();
		// modelTemperature[5] = (int) entity.getPeriodMarB();
		// modelTemperature[6] = (int) entity.getPeriodAprA();
		// modelTemperature[7] = (int) entity.getPeriodAprB();
		// modelTemperature[8] = (int) entity.getPeriodMayA();
		// modelTemperature[9] = (int) entity.getPeriodMayB();
		// modelTemperature[10] = (int) entity.getPeriodJunA();
		// modelTemperature[11] = (int) entity.getPeriodJunB();
		// modelTemperature[12] = (int) entity.getPeriodJulA();
		// modelTemperature[13] = (int) entity.getPeriodJulB();
		// modelTemperature[14] = (int) entity.getPeriodAugA();
		// modelTemperature[15] = (int) entity.getPeriodAugB();
		// modelTemperature[16] = (int) entity.getPeriodSepA();
		// modelTemperature[17] = (int) entity.getPeriodSepB();
		// modelTemperature[18] = (int) entity.getPeriodOctA();
		// modelTemperature[19] = (int) entity.getPeriodOctB();
		// modelTemperature[20] = (int) entity.getPeriodNovA();
		// modelTemperature[21] = (int) entity.getPeriodNovB();
		// modelTemperature[22] = (int) entity.getPeriodDecA();
		// modelTemperature[23] = (int) entity.getPeriodDecB();
		// } catch (Exception e) {
		// // TODO Auto-generated catch block
		// e.printStackTrace();
		// }
		//
		// // temperature
		// Calendar startDate = Calendar.getInstance();
		// startDate.setTime(mScenario.getStartDate());
		// startDate.set(Calendar.HOUR, 0);
		// startDate.set(Calendar.MINUTE, 0);
		// startDate.set(Calendar.SECOND, 0);
		// startDate.set(Calendar.MILLISECOND, 0);
		// Calendar endDate = Calendar.getInstance();
		// endDate.setTime(mScenario.getTargetDate());
		// endDate.set(Calendar.HOUR, 0);
		// endDate.set(Calendar.MINUTE, 0);
		// endDate.set(Calendar.SECOND, 0);
		// endDate.set(Calendar.MILLISECOND, 0);
		// for (Calendar curDate = startDate; !curDate.after(endDate);
		// curDate.add(Calendar.DAY_OF_MONTH, 1))
		//
		// {
		// int month = curDate.get(Calendar.MONTH) + 1;
		// int dayMonth = curDate.get(Calendar.DAY_OF_MONTH);
		// int idx = ((2 * month) - (dayMonth >= 15 ? 0 : 1)) - 1;
		// temperatureTable.put(curDate.getTimeInMillis(),
		// modelTemperature[idx]);
		// }
		// if (logger.isTraceEnabled())
		// logger.trace(String.format("temperature table from db [%s]",
		// temperatureTable));
		//
		// // load scenario from db
		//
		// // FCR
		// List<Fcr> fcrs = new ArrayList<>();
		//
		// {
		// try {
		// fcrs.addAll(FcrLocalServiceUtil.getFcrs(mScenario.getSimulModelId()));
		//
		// } catch (SystemException e) {
		// // TODO Auto-generated catch block
		// e.printStackTrace();
		// }
		// if (logger.isTraceEnabled())
		// logger.trace(String.format("fcr list from db [%s]", fcrs));
		// fcrs = fcrByTempWeight.sortedCopy(fcrs);
		// if (logger.isTraceEnabled())
		// logger.trace(String.format("fcr list from db sorted [%s]", fcrs));
		// double lastLimit = 0;
		// double prevMab = Double.MAX_VALUE;
		// int curTemp = -1;
		// RangeMap<Double, Double> tempColumn = null;
		// // should be sorted to temperature asc, then weight desc
		// for (Fcr fcr : fcrs) {
		// if (fcr.getTemperature() != curTemp) {
		// // temperature value changed; save previous
		// if (tempColumn != null) {
		// // take care of the lower limit
		// if (lastLimit > 0) {
		// tempColumn.put(Range.closedOpen(0.0, lastLimit), 0.0);
		// }
		// fcrTable.put(curTemp, tempColumn);
		// }
		//
		// curTemp = fcr.getTemperature();
		// // reset
		// tempColumn = TreeRangeMap.create();
		// prevMab = Double.MAX_VALUE;
		// }
		// double mab = fcr.getFromWeight();
		// double fcrValue = fcr.getValue() / 100.0;
		// tempColumn.put(Range.closedOpen(mab, prevMab), fcrValue);
		// lastLimit = mab;
		// prevMab = mab;
		// }
		// // last temperature didn't get a chance to see a temperature value
		// // change
		// if (tempColumn != null) {
		// if (lastLimit > 0) {
		// tempColumn.put(Range.closedOpen(0.0, lastLimit), 0.0);
		// }
		// fcrTable.put(curTemp, tempColumn);
		// }
		// }
		// if (logger.isTraceEnabled())
		// logger.trace(String.format("fcr table [%s]", fcrTable));
		//
		// // SFR
		// List<Sfr> sfrs = new ArrayList<>();
		//
		// {
		// try {
		// sfrs.addAll(SfrLocalServiceUtil.getSfrs(mScenario.getSimulModelId()));
		//
		// } catch (SystemException e) {
		// // TODO Auto-generated catch block
		// e.printStackTrace();
		// }
		// if (logger.isTraceEnabled())
		// logger.trace(String.format("sfr list from db [%s]", sfrs));
		// sfrs = sfrByTempWeight.sortedCopy(sfrs);
		// if (logger.isTraceEnabled())
		// logger.trace(String.format("sfr list from db sorted [%s]", sfrs));
		// double lastLimit = 0;
		// double prevMab = Double.MAX_VALUE;
		// int curTemp = -1;
		// RangeMap<Double, Double> tempColumn = null;
		// // should be sorted to temperature asc, then weight desc
		// for (Sfr sfr : sfrs) {
		// if (sfr.getTemperature() != curTemp) {
		// // temperature value changed; save previous
		// if (tempColumn != null) {
		// // take care of the lower limit
		// if (lastLimit > 0) {
		// tempColumn.put(Range.closedOpen(0.0, lastLimit), 0.0);
		// }
		// sfrTable.put(curTemp, tempColumn);
		// }
		//
		// curTemp = sfr.getTemperature();
		// // reset
		// tempColumn = TreeRangeMap.create();
		// prevMab = Double.MAX_VALUE;
		// }
		// double mab = sfr.getFromWeight();
		// double sfrValue = sfr.getValue() / 100.0;
		// tempColumn.put(Range.closedOpen(mab, prevMab), sfrValue);
		// lastLimit = mab;
		// prevMab = mab;
		// }
		// // last temperature didn't get a chance to see a temperature value
		// // change
		// if (tempColumn != null) {
		// if (lastLimit > 0) {
		// tempColumn.put(Range.closedOpen(0.0, lastLimit), 0.0);
		// }
		// sfrTable.put(curTemp, tempColumn);
		// }
		// }
		// if (logger.isTraceEnabled())
		// logger.trace(String.format("sfr table [%s]", sfrTable));
		//
		// // Mortality
		// List<Mortality> mortalities = new ArrayList<>();
		//
		// {
		// try {
		// mortalities.addAll(MortalityLocalServiceUtil.getMortalities(mScenario.getSimulModelId()));
		//
		// } catch (SystemException e) {
		// // TODO Auto-generated catch block
		// e.printStackTrace();
		// }
		// if (logger.isTraceEnabled())
		// logger.trace(String.format("mortality list from db [%s]",
		// mortalities));
		// mortalities = mortalityByTempWeight.sortedCopy(mortalities);
		// if (logger.isTraceEnabled())
		// logger.trace(String.format("mortality list from db sorted [%s]",
		// mortalities));
		// double lastLimit = 0;
		// double prevMab = Double.MAX_VALUE;
		// int curTemp = -1;
		// RangeMap<Double, Double> tempColumn = null;
		// // should be sorted to temperature asc, then weight desc
		// for (Mortality mortality : mortalities) {
		// if (mortality.getTemperature() != curTemp) {
		// // temperature value changed; save previous
		// if (tempColumn != null) {
		// // take care of the lower limit
		// if (lastLimit > 0) {
		// tempColumn.put(Range.closedOpen(0.0, lastLimit), 0.0);
		// }
		// mortalityTable.put(curTemp, tempColumn);
		// }
		//
		// curTemp = mortality.getTemperature();
		// // reset
		// tempColumn = TreeRangeMap.create();
		// prevMab = Double.MAX_VALUE;
		// }
		// double mab = mortality.getFromWeight();
		// double mortalityValue = mortality.getValue() / 100.0;
		// tempColumn.put(Range.closedOpen(mab, prevMab), mortalityValue);
		// lastLimit = mab;
		// prevMab = mab;
		// }
		// // last temperature didn't get a chance to see a temperature value
		// // change
		// if (tempColumn != null) {
		// if (lastLimit > 0) {
		// tempColumn.put(Range.closedOpen(0.0, lastLimit), 0.0);
		// }
		// mortalityTable.put(curTemp, tempColumn);
		// }
		// }
		// if (logger.isTraceEnabled())
		// logger.trace(String.format("mortality table [%s]", mortalityTable));
		//
		// long duration = System.currentTimeMillis() - start;
		// if (logger.isTraceEnabled())
		// logger.trace(String.format("preparation took [%s] ms", duration));
		//
	}

	private void afterRun() {
		// try {
		// if (logger.isDebugEnabled())
		// logger.debug(String.format("updating [%s]", mScenario));
		// ScenarioLocalServiceUtil.updateScenario(mScenario);
		// } catch (SystemException e) {
		// // TODO Auto-generated catch block
		// e.printStackTrace();
		// }
	}

	@Override
	public String toString() {
		return MoreObjects.toStringHelper(this).add("logger", logger).add("mScenario", mScenario)
				.add("fcrTable", fcrTable).add("sfrTable", sfrTable).add("mortalityTable", mortalityTable)
				.add("temperatureTable", temperatureTable)
				.add("modelTemperature", Arrays.deepToString(new Object[] { modelTemperature }))
				.add("dailyResults", Arrays.deepToString(new Object[] { dailyResults })).toString();
	}

	private void doRun() {
		// TODO Auto-generated method stub
		long start = System.currentTimeMillis();

		Calendar startDate = Calendar.getInstance();
		startDate.setTime(mScenario.getStartDate());
		startDate.set(Calendar.HOUR, 0);
		startDate.set(Calendar.MINUTE, 0);
		startDate.set(Calendar.SECOND, 0);
		startDate.set(Calendar.MILLISECOND, 0);
		Calendar endDate = Calendar.getInstance();
		endDate.setTime(mScenario.getTargetDate());
		endDate.set(Calendar.HOUR, 0);
		endDate.set(Calendar.MINUTE, 0);
		endDate.set(Calendar.SECOND, 0);
		endDate.set(Calendar.MILLISECOND, 0);

		double startBM = mScenario.getWeight() * mScenario.getFishNo();

		double totalFood = 0;
		double totalDeadBM = 0.0;
		int totalDead = 0;
		if (logger.isTraceEnabled())
			logger.trace(String.format("temperatureTable %s", temperatureTable));
		if (logger.isTraceEnabled())
			logger.trace(String.format("sfrTable %s", sfrTable));
		if (logger.isTraceEnabled())
			logger.trace(String.format("fcrTable %s", fcrTable));

		Daily curDay = new Daily();
		// startup
		curDay.mab = mScenario.getWeight();
		curDay.fishcount = mScenario.getFishNo();

		for (Calendar curDate = startDate; !curDate.after(endDate); curDate.add(Calendar.DAY_OF_MONTH, 1)) {
			if (logger.isTraceEnabled())
				logger.trace(String.format("on day [%s] as millis [%s]", FULL_DATE_FORMAT.format(curDate.getTime()),
						curDate.getTimeInMillis()));
			// prepare the day
			curDay.date.setTimeInMillis(curDate.getTimeInMillis());
			curDay.dead = 0;
			curDay.food = 0.0;
			curDay.growth = 0.0;
			// get indicators
			Integer temperature = temperatureTable.get(curDay.date.getTimeInMillis());
			if (logger.isTraceEnabled())
				logger.trace(String.format("temperature %s", temperature));
			if (logger.isTraceEnabled())
				logger.trace(
						String.format("sfr temperature [%s] selection %s", temperature, sfrTable.get(temperature)));
			curDay.temperature = temperature;
			Double sfr = sfrTable.get(temperature).get(curDay.mab);
			if (logger.isTraceEnabled())
				logger.trace(String.format("sfr on mab [%s] selection %s", curDay.mab, sfr));
			if (logger.isTraceEnabled())
				logger.trace(
						String.format("fcr temperature [%s] selection %s", temperature, fcrTable.get(temperature)));
			curDay.sfr = sfr * 100;
			Double fcr = fcrTable.get(temperature).get(curDay.mab);
			if (logger.isTraceEnabled())
				logger.trace(String.format("fcr on mab [%s] selection %s", curDay.mab, fcr));
			curDay.fcr = fcr * 100;
			// start calc
			curDay.dead = (int) (mortalityTable.get(temperature).get(curDay.mab) * curDay.fishcount);
			curDay.deadBM = curDay.dead * curDay.mab;
			totalDead += curDay.dead;
			totalDeadBM += curDay.deadBM;
			curDay.fishcount -= curDay.dead;
			curDay.food = (curDay.mab * curDay.fishcount) * sfr;
			totalFood += curDay.food;
			double foodPerFish = curDay.food / curDay.fishcount;
			curDay.growth = fcr > 0 ? foodPerFish / (fcr * 100.0) : 0;
			if (logger.isTraceEnabled())
				logger.trace(String.format("adding daily [%s]", curDay));
			// day ends here
			double dayBM = curDay.mab * curDay.fishcount;
			Double dBM = dayBM - startBM; // for the economical fcr
			curDay.fcrEcon = (dBM > 0 ? totalFood / dBM : 0);
			dBM = dBM - totalDeadBM; // for the biological fcr
			curDay.fcrBiol = (dBM > 0 ? totalFood / dBM : 0);
			dailyResults.add(curDay);

			// prepare next day
			Daily prevDay = curDay;
			curDay = new Daily(prevDay);
			curDay.mab += prevDay.growth;
			if (logger.isTraceEnabled())
				logger.trace(String.format("prevDay [%s] curDay [%s]", prevDay, curDay));
			prevDay = null; // facilitate garbage collector
		}
		if (logger.isDebugEnabled())
			logger.debug(String.format("final day [%s]", curDay));
		if (logger.isTraceEnabled())
			logger.trace(String.format("detailed daily results [%s]", dailyResults));

		double finalBM = curDay.mab * curDay.fishcount;
		// TODO rename weight to BM
		mScenario.setResultsWeight(curDay.mab);
		Double dBMEcon = finalBM - startBM;
		mScenario.setResultsEconFCR(dBMEcon > 0 ? totalFood / dBMEcon : 0);
		Double dBMBiol = dBMEcon - totalDeadBM;
		mScenario.setResultsBiolFCR(dBMBiol > 0 ? totalFood / dBMBiol : 0);
		mScenario.setResultsMortality(100 * ((double) totalDead / (double) mScenario.getFishNo()));

		// Graphs
		SimpleDateFormat dateFormat = new SimpleDateFormat("dd/MM/yy");
		String tableWeight = new String();
		String tableFCR = new String();
		String tableFood = new String();
		String recordWeight = "{c:[{v: '%s'}, {v: %.2f}]},";
		String recordFCR = "{c:[{v: '%s'}, {v: %.2f}, {v: %.2f}]},";
		String recordFood = "{c:[{v: '%s-%s'}, {v: %.2f}]},";
		int foodMonth = -1;
		int foodYear = -1;
		double foodCons = 0;
		for (Daily daily : dailyResults) {
			String day = dateFormat.format(daily.date.getTime());
			tableWeight = tableWeight + String.format(recordWeight, day, daily.mab);
			// TODO a/ which fcr I need? and b/ the 2nd fcr should be the global
			// fcr
			if (daily.fcrBiol > 0)
				tableFCR = tableFCR + String.format(recordFCR, day, daily.fcrBiol, daily.fcrEcon);
			// food consumption (monthly)
			if (foodMonth != daily.date.get(Calendar.MONTH) || foodYear != daily.date.get(Calendar.YEAR)) {
				// save and restart
				if (foodCons > 0) {
					tableFood = tableFood + String.format(recordFood, foodMonth + 1, foodYear, foodCons / 1000.0);
				}
				foodCons = 0;
				foodMonth = daily.date.get(Calendar.MONTH);
				foodYear = daily.date.get(Calendar.YEAR);
			}
			foodCons += curDay.food;
		}
		// take care of possible orphan acc values
		if (foodCons > 0) {
			tableFood = tableFood + String.format(recordFood, foodMonth + 1, foodYear, foodCons / 1000.0);
		}

		mScenario.setResultsGraphData(tableWeight + "gri2sbbridge" + tableFCR + "gri2sbbridge" + tableFood);

		long duration = System.currentTimeMillis() - start;
		if (logger.isDebugEnabled())
			logger.debug(String.format("actual run took [%s] ms", duration));

		if (logger.isDebugEnabled())
			logger.debug(String.format("results [%s]", mScenario));
	}

}
