package com.finconsgroup.itserr.marketplace.zenodo.bs.zenodo;

import com.finconsgroup.itserr.marketplace.zenodo.bs.client.zenodo.ZenodoClient;
import com.finconsgroup.itserr.marketplace.zenodo.bs.client.zenodo.dto.InputOauthTokenRequestDto;
import com.finconsgroup.itserr.marketplace.zenodo.bs.client.zenodo.dto.OutputOauthTokenDto;
import com.finconsgroup.itserr.marketplace.zenodo.bs.config.properties.ZenodoConfigurationProperties;
import com.finconsgroup.itserr.marketplace.zenodo.bs.config.properties.ZenodoOauthProperties;
import com.finconsgroup.itserr.marketplace.zenodo.bs.exception.ZenodoAuthFailureException;
import lombok.RequiredArgsConstructor;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.net.URI;

/**
 * Service for authenticating with the Zenodo API. This class provides functionality to exchange an authorization code for an OAuth token using the Zenodo API.
 * <p>
 * It requires the {@code ZenodoConfigurationProperties} for accessing OAuth configuration details and the {@code ZenodoClient} to make requests to the Zenodo
 * API endpoints.
 */
@Component
@RequiredArgsConstructor
public class ZenodoAuthenticator {

    /** Configuration properties for Zenodo integration */
    private final ZenodoConfigurationProperties config;

    /** Client for making API calls to Zenodo service */
    private final ZenodoClient zenodoClient;

    /**
     * Validates the provided authorization code and exchanges it for an OAuth token.
     *
     * @param authCode the authorization code to validate
     * @param redirectUri the redirect URI used to obtain the authorization code from Zenodo
     * @return OutputOauthTokenDto containing the OAuth token details
     * @throws ZenodoAuthFailureException if authentication with Zenodo service fails
     */
    @NonNull
    public OutputOauthTokenDto getToken(
            @NonNull final String authCode,
            @NonNull final URI redirectUri) {

        ZenodoOauthProperties oauth = config.getOauth();

        try {

            return zenodoClient.getToken(
                    InputOauthTokenRequestDto.builder()
                            .client_id(oauth.getClientId())
                            .client_secret(oauth.getClientSecret())
                            .grant_type("authorization_code")
                            .code(authCode)
                            .redirect_uri(redirectUri.toString())
                            .build());

        } catch (final Exception e) {

            throw new ZenodoAuthFailureException(e);

        }

    }

}
