package com.finconsgroup.itserr.marketplace.zenodo.bs.mapper;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.zenodo.bs.messaging.dto.ExportRequestCreatedMessageBodyDto;
import com.finconsgroup.itserr.marketplace.zenodo.bs.validation.CreateExportRequestValidator;
import com.finconsgroup.itserr.messaging.dto.MessagingEventUserDto;
import org.mapstruct.Mapper;
import org.mapstruct.Mapping;

/**
 * Mapper interface for converting export requests.
 */
@Mapper(config = MapperConfiguration.class)
public interface ExportRequestMapper {

    /**
     * Converts a validated export request to a message DTO.
     *
     * @param src The validated export request to convert
     * @return The converted message DTO
     */
    @Mapping(target = "id", source = "request.source.id")
    @Mapping(target = "user", expression = "java(this.getMessageUserFromSecurity())")
    @Mapping(target = "name", ignore = true)
    @Mapping(target = "title", source = "catalogItem.title")
    @Mapping(target = "status", ignore = true)
    @Mapping(target = "category", ignore = true)
    @Mapping(target = "message", ignore = true)
    @Mapping(target = "timestamp", expression = "java(java.time.Instant.now())")
    @Mapping(target = "additionalData", expression = "java(this.toExportRequestCreateAdditionalDataDto(src))")
    ExportRequestCreatedMessageBodyDto toMessageDto(
            CreateExportRequestValidator.ValidatedRequest src);

    /**
     * Creates a messaging event user DTO from the current security context.
     *
     * @return The messaging event user DTO
     */
    default MessagingEventUserDto getMessageUserFromSecurity() {
        return MessagingEventUserDto.builder()
                .id(JwtTokenHolder.getUserId().orElse(null))
                .username(JwtTokenHolder.getPreferredUsername().orElse(null))
                .name(JwtTokenHolder.getName().orElse(null))
                .build();
    }

    /**
     * Converts a validated export request to additional data DTO.
     *
     * @param src The validated export request to convert
     * @return The additional data DTO
     */
    @Mapping(target = "sourceType", source = "request.source.type")
    @Mapping(target = "d4sJwtToken", expression = "java(com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder.getToken())")
    @Mapping(target = "zenodoJwtToken", source = "zenodoToken.accessToken")
    @Mapping(target = "zenodoJwtRefreshToken", source = "zenodoToken.refreshToken")
    ExportRequestCreatedMessageBodyDto.AdditionalData toExportRequestCreateAdditionalDataDto(
            CreateExportRequestValidator.ValidatedRequest src);

}
