package com.finconsgroup.itserr.marketplace.zenodo.bs.validation;

import com.finconsgroup.itserr.marketplace.zenodo.bs.client.catalog.dto.OutputCatalogItemDto;
import com.finconsgroup.itserr.marketplace.zenodo.bs.client.zenodo.dto.OutputOauthTokenDto;
import com.finconsgroup.itserr.marketplace.zenodo.bs.dto.InputCreateZenodoExportDto;
import com.finconsgroup.itserr.marketplace.zenodo.bs.exception.NoSourceItemIdException;
import com.finconsgroup.itserr.marketplace.zenodo.bs.exception.NoZenodoAuthCodeException;
import com.finconsgroup.itserr.marketplace.zenodo.bs.exception.NoZenodoRedirectUriException;
import com.finconsgroup.itserr.marketplace.zenodo.bs.zenodo.ZenodoAuthenticator;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.net.URI;
import java.util.Optional;

/**
 * Validator component responsible for validating Zenodo export requests, including authentication and catalog item validation.
 */
@Component
@RequiredArgsConstructor
public class CreateExportRequestValidator {

    /** Authenticator component for Zenodo service authentication */
    private final ZenodoAuthenticator zenodoAuthenticator;

    /** Validator component for catalog item validation */
    private final CatalogItemValidator catalogItemValidator;

    /**
     * <p>Record representing a validated Zenodo export request. Contains the original request details,
     * the authenticated Zenodo token, and the validated catalog item.</p>
     *
     * <p>The `ValidatedRequest` is created after successfully validating the input export request,
     * ensuring that the user is authenticated with Zenodo and the requested catalog item exists and is authorized.</p>
     */
    public record ValidatedRequest(
            InputCreateZenodoExportDto request,
            OutputOauthTokenDto zenodoToken,
            OutputCatalogItemDto catalogItem ) {
    }

    /**
     * Validates a Zenodo export request by checking authentication and catalog item existence.
     *
     * @param request the export request to validate
     * @return ValidatedRequest containing the validated request details, the Zenodo tokens and the catalog item
     * @throws NoZenodoAuthCodeException if the Zenodo authentication code is missing
     * @throws NoSourceItemIdException if the source catalog item ID is missing
     */
    public ValidatedRequest validate(final InputCreateZenodoExportDto request) {

        // Validate request
        final InputCreateZenodoExportDto.ZenodoDto zenodo = Optional.ofNullable(request)
                .map(InputCreateZenodoExportDto::getZenodo)
                .orElseThrow(NoZenodoAuthCodeException::new);
        final String authCode = Optional.of(zenodo)
                .map(InputCreateZenodoExportDto.ZenodoDto::getAuthCode)
                .filter(StringUtils::isNotBlank)
                .orElseThrow(NoZenodoAuthCodeException::new);
        final URI redirectUri = Optional.of(zenodo)
                .map(InputCreateZenodoExportDto.ZenodoDto::getRedirectUri)
                .orElseThrow(NoZenodoRedirectUriException::new);
        final String catalogItemId = Optional.of(request)
                .map(InputCreateZenodoExportDto::getSource)
                .map(InputCreateZenodoExportDto.SourceDto::getId)
                .filter(StringUtils::isNotBlank)
                .orElseThrow(NoSourceItemIdException::new);

        // Validate auth code and get JWT token
        final OutputOauthTokenDto token = zenodoAuthenticator.getToken(authCode, redirectUri);

        // Validate catalog item
        final OutputCatalogItemDto catalogItem = catalogItemValidator.validate(catalogItemId);

        // Return the validated result
        return new ValidatedRequest(request, token, catalogItem);

    }

}
