package com.finconsgroup.itserr.marketplace.zenodo.bs.dto;

import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.v3.oas.annotations.media.Schema;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotBlank;
import jakarta.validation.constraints.NotNull;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.net.URI;

/**
 * Data transfer object for creating a new Zenodo export request. Contains source information and Zenodo authentication details.
 */
@Schema(
        name = "CreateZenodoExportDto",
        description = "Request object for creating a new Zenodo export containing source and authentication details")
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class InputCreateZenodoExportDto {

    /**
     * Enum representing the type of source for Zenodo export. Defines the possible types of sources that can be exported to Zenodo.
     */
    @Schema(name = "ZenodoExportSourceType")
    public enum SourceType {

        /**
         * Represents a catalog item source type for export
         */
        @JsonProperty("catalog-item")
        CATALOG_ITEM

    }

    /**
     * Inner class representing the source information for the Zenodo export.
     */
    @Schema(
            name = "ZenodoExportSourceDto",
            description = "Source information for the Zenodo export")
    @Data
    @Builder
    @NoArgsConstructor
    @AllArgsConstructor
    public static class SourceDto {

        /** The type of source for the export request */
        @NotNull
        @Schema(description = "Type of source for the export request", example = "catalog-item")
        private SourceType type;

        /** The unique identifier of the source item */
        @NotBlank
        @Schema(description = "id of the item to be exported to Zenodo", example = "my-catalog-item")
        private String id;

    }

    /**
     * Inner class containing Zenodo-specific authentication information.
     */
    @Schema(description = "Zenodo-specific authentication information")
    @Data
    @Builder
    @NoArgsConstructor
    @AllArgsConstructor
    public static class ZenodoDto {

        /** The authentication code for Zenodo API access */
        @NotBlank
        @Schema(description = "Authentication code for Zenodo API access", example = "123456789")
        private String authCode;

        /** The redirect URI used to get the authentication code from Zenodo. */
        @NotNull
        @Schema(description = "The redirect URI used to get the authentication code from Zenodo", example = "https://my-host/auth-code")
        private URI redirectUri;

    }

    /** The source information containing type and ID for the export */
    @Valid
    @NotNull
    @Schema(description = "Source information containing type and ID for the export")
    private SourceDto source;

    /** The Zenodo authentication code obtained using the OAUTH authorization_code flow on Zenodo */
    @Valid
    @NotNull
    @Schema(description = "Zenodo authentication details")
    private ZenodoDto zenodo;

}
