package com.finconsgroup.itserr.marketplace.userprofile.dm.dto;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2AuthenticationException;

import java.util.UUID;

/**
 * Represents an authenticated user's core identity information extracted from a JWT token.
 *
 * <p><b>Usage Example:</b></p>
 * <pre>{@code
 * AuthenticatedUser user = AuthenticatedUser.fromJwt();
 * log.info("User ID: {}", user.userId());
 * }</pre>
 *
 * @see JwtTokenHolder
 */
public record AuthenticatedUser(
        UUID userId,
        String firstName,
        String lastName,
        String email,
        String preferredUsername
) {
    /**
     * Creates an {@code AuthenticatedUser} instance by extracting claims from the current JWT token.
     * <p>
     * This factory method ensures all required claims are present and valid. If any claim is missing,
     * the underlying {@link JwtTokenHolder} will throw an exception.
     * </p>
     *
     * @return a new {@code AuthenticatedUser} populated with JWT claims
     * @throws WP2AuthenticationException if any required claim is missing or invalid
     */
    public static AuthenticatedUser fromJwt() {
        return new AuthenticatedUser(
                JwtTokenHolder.getUserIdOrThrow(),
                JwtTokenHolder.getGivenNameOrThrow(),
                JwtTokenHolder.getFamilyNameOrThrow(),
                JwtTokenHolder.getEmailOrThrow(),
                JwtTokenHolder.getPreferredUsernameOrThrow()
        );
    }
}