package com.finconsgroup.itserr.marketplace.userprofile.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.OneToMany;
import jakarta.persistence.OrderBy;
import jakarta.persistence.Table;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.ToString;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.util.List;
import java.util.UUID;

@Entity
@Table(name = "archived_expertise")
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ArchivedExpertiseEntity extends AbstractUUIDEntity {

    @EqualsAndHashCode.Exclude
    @ManyToOne
    @JoinColumn(name = "user_profile_id", nullable = false)
    private UserProfileEntity userProfile;

    /**
     * The unique identifier of the label.
     */
    @Column(name = "label_id")
    private UUID labelId;

    /**
     * The display name or name of the expertise.
     */
    @Column(name = "display_name")
    private String displayName;

    /**
     * The total number of endorsements received for this expertise.
     */
    @Column(name = "endorsement_count")
    private Integer endorsementCount;

    /**
     * The list of endorsements associated with this expertise.
     */
    @OneToMany(cascade = CascadeType.ALL, orphanRemoval = true, mappedBy = "expertise")
    @OrderBy("creationTime ASC")
    @ToString.Exclude
    private List<ArchivedEndorsementEntity> endorsements;

    /**
     * The ordering column.
     */
    @Column(name = "expertise_order", nullable = false)
    private long expertiseOrder;

    /**
     * The timestamp when the event was created.
     */
    @Column(name = "creation_time", nullable = false)
    private Instant creationTime;

    /**
     * The timestamp when the event was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The timestamp when the event was last archived.
     */
    @Column(name = "archive_time", nullable = false)
    private Instant archiveTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     */
    @Column(name = "version", nullable = false)
    private long version;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
        if (archiveTime == null) {
            archiveTime = now;
        }
    }

}
