package com.finconsgroup.itserr.marketplace.userprofile.dm.service;

import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.AuthenticatedUser;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputAddProjectToUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByIdsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByPrincipalsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputFindUserProfilesByTokenInfoDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputPatchUserProfileProjectDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputRemoveProjectFromUserProfilesDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.InputUpdateUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.dto.OutputUserProfileFolderDetailsDto;
import com.finconsgroup.itserr.marketplace.userprofile.dm.exception.UserProfileNotFoundException;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;

import java.util.List;
import java.util.UUID;

/**
 * Service for handling business logic related to UserProfiles.
 * This interface provides methods to retrieve, create, update, and delete UserProfiles,
 * as well as retrieve UserProfiles by category.
 *
 * <p>Example usage:
 * <pre>
 * Page<OutputUserProfileDto> UserProfiles = userprofileService.findAll(pageable);
 * </pre>
 * </p>
 */
public interface UserProfileService {

    /**
     * Creates a new userprofile using the provided input data.
     *
     * @param inputCreateUserProfileDto the input DTO containing userprofile details
     * @return an OutputUserProfileDto representing the created userprofile
     */
    @NonNull
    OutputUserProfileDto create(@NonNull AuthenticatedUser authenticatedUser,
                                @NonNull InputUserProfileDto inputCreateUserProfileDto);

    /**
     * Retrieves a paginated list of all UserProfiles.
     *
     * @param pageable the pagination information
     * @return a page of OutputUserProfileDto representing the UserProfiles
     */
    @NonNull
    Page<OutputUserProfileDto> findAll(@NonNull Pageable pageable);

    /**
     * Search for documents for the provided title string which can contain multiple terms.
     * It returns the results in suitable format that can be used for autocompletion.
     *
     * @param terms    the terms to search for
     * @param pageable the pagination information
     * @return Page of {@link OutputUserProfileAutoCompleteDto} of matching documents
     */
    @NonNull
    Page<OutputUserProfileAutoCompleteDto> getAutoCompletions(@NonNull String terms, @NonNull Pageable pageable);

    /**
     * Retrieves a userprofile by id.
     *
     * @param authenticatedUser the contains the id of the userprofile to retrieve
     * @return an OutputUserProfileDto representing the userprofile details
     * @throws UserProfileNotFoundException if no userprofile is found with the given id
     */
    @NonNull
    OutputUserProfileDto findById(@NonNull AuthenticatedUser authenticatedUser);

    /**
     * Retrieves a userprofile by profileId.
     *
     * @param profileId the contains the profileId of the userprofile to retrieve
     * @return an OutputUserProfileDto representing the userprofile details
     * @throws UserProfileNotFoundException if no userprofile is found for the given id
     */
    @NonNull
    OutputUserProfileDto getById(@NonNull UUID profileId);

    /**
     * Retrieves a paginated list of user profiles by ids.
     *
     * @param inputFindUserProfilesByIdsDto the input DTO containing ids to retrieve details for
     * @param pageable                      the pagination information
     * @return a page of OutputUserProfileDto representing the userprofile details
     */
    @NonNull
    Page<OutputUserProfileDto> findAllByIds(@NonNull InputFindUserProfilesByIdsDto inputFindUserProfilesByIdsDto,
                                            @NonNull Pageable pageable);

    /**
     * Updates an existing userprofile by id with the provided data.
     *
     * @param userId                    the id of the userprofile to update
     * @param inputUpdateUserProfileDto the input DTO containing updated userprofile details
     * @return an OutputUserProfileDto representing the updated userprofile
     * @throws UserProfileNotFoundException if no userprofile is found with the given id
     */
    @NonNull
    OutputUserProfileDto updateById(UUID userId, @NonNull InputUpdateUserProfileDto inputUpdateUserProfileDto);

    /**
     * Retrieves the folder details of userprofile by id.
     *
     * @param userId the id of the userprofile to update
     * @return an OutputUserProfileFolderDetails representing the details of userprofile folder
     * @throws UserProfileNotFoundException if no userprofile is found with the given id
     */
    @NonNull
    OutputUserProfileFolderDetailsDto getUserProfileFolderId(UUID userId);

    /**
     * Patch the info of an user.
     *
     * @param patchUserProfileDto the user profile info to be patched
     * @return an OutputPatchUserProfileDto representing the details patched on th user
     */
    @NonNull
    OutputPatchUserProfileDto patchUserProfileInfo(InputPatchUserProfileDto patchUserProfileDto);

    /**
     * Retrieves a paginated list of user profiles by their principals.
     *
     * @param inputFindUserProfilesByPrincipalsDto the input DTO containing principals to retrieve details for
     * @param pageable                             the pagination information
     * @return a page of OutputUserProfileDto representing the userprofile details
     */
    @NonNull
    Page<OutputUserProfileDto> findAllByPrincipals(InputFindUserProfilesByPrincipalsDto inputFindUserProfilesByPrincipalsDto, Pageable pageable);

    /**
     * add projects to existing userprofile for given ids.
     *
     * @param inputAddProjectToUserProfilesDto contains the list of userIds and object containing updated userprofile project details
     * @return the updated {@link List<OutputUserProfileDto>} and HTTP status 200 (OK)
     */
    @NonNull
    List<OutputUserProfileDto> addProjectToUserProfiles(@NonNull InputAddProjectToUserProfilesDto inputAddProjectToUserProfilesDto);

    /**
     * removes projects from existing userprofile for given ids.
     *
     * @param inputRemoveProjectFromUserProfilesDto contains the list of userIds and object containing updated userprofile project details
     * @return the updated {@link List<OutputUserProfileDto>} and HTTP status 200 (OK)
     */
    @NonNull
    List<OutputUserProfileDto> removeProjectFromUserProfiles(@NonNull InputRemoveProjectFromUserProfilesDto inputRemoveProjectFromUserProfilesDto);

    /**
     * update project for existing userprofile for given ids.
     *
     * @param inputPatchUserProfileProjectDto contains the list of userIds and patch updates for userprofile project details
     * @return the updated {@link List<OutputUserProfileDto>} and HTTP status 200 (OK)
     */
    @NonNull
    List<OutputUserProfileDto> patchUserProfileProject(@NonNull InputPatchUserProfileProjectDto inputPatchUserProfileProjectDto);

    /**
     * Retrieves a list of user profiles by a mix of token info.
     *
     * @param inputFindUserProfilesByTokenInfoDto the input DTO containing principals to retrieve details for
     * @return a page of OutputUserProfileDto representing the userprofile details
     */
    List<OutputUserProfileDto> findAllByTokenInfo(InputFindUserProfilesByTokenInfoDto inputFindUserProfilesByTokenInfoDto);

    /**
     * activate/deactivate a user profile.
     */
    @NonNull
    void processUserProfileStatusChange(@NonNull UUID userId, @NonNull Boolean active);

    /**
     * Retrieves the list of user profile ids whose interests match at least one the provided list of strings.
     *
     * @param stringsToCheck the strings to match interests against
     * @return the matching {@link List} of user profile ids
     */
    List<UUID> findMatchingInterests(@NonNull List<String> stringsToCheck);
}
