package com.finconsgroup.itserr.marketplace.userprofile.dm.enums;

import lombok.Getter;

/**
 * Represents the available theme preferences for a user's interface.
 * <p>
 * The theme determines the color scheme or visual style applied to the application's UI.
 * Supported types include:
 * <ul>
 *     <li>{@link #LIGHT} — A light-colored theme suitable for bright environments.</li>
 *     <li>{@link #DARK} — A dark-colored theme suitable for low-light or night environments.</li>
 *     <li>{@link #SYSTEM} — Automatically matches the system's current theme preference.</li>
 * </ul>
 */
@Getter
public enum ThemeType {

    /**
     * Light theme mode.
     */
    LIGHT("light"),

    /**
     * Dark theme mode.
     */
    DARK("dark"),

    /**
     * Theme mode that follows the system setting.
     */
    SYSTEM("system");

    /**
     * The string value representing the theme type.
     */
    private final String value;

    /**
     * Constructs a {@code ThemeType} with the specified string value.
     *
     * @param value the string representation of the theme type
     */
    ThemeType(String value) {
        this.value = value;
    }

    /**
     * Returns the {@code ThemeType} corresponding to the given string value.
     *
     * @param value the string value representing a theme type
     * @return the matching {@code ThemeType}
     * @throws IllegalArgumentException if the specified value does not match any known theme type
     */
    public static ThemeType fromValue(String value) {
        for (ThemeType type : ThemeType.values()) {
            if (type.value.equalsIgnoreCase(value)) { // allows case-insensitive match
                return type;
            }
        }
        throw new IllegalArgumentException("Unknown theme type: " + value);
    }

    /**
     * Returns the string value representation of this theme type.
     *
     * @return the string value of this theme type
     */
    @Override
    public String toString() {
        return value;
    }
}
