package com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.impl;

import com.finconsgroup.itserr.marketplace.userprofile.bs.config.properties.UserProfileBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.EventProducer;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.exception.WP2MessagingException;
import com.finconsgroup.itserr.messaging.producer.MessageProducer;
import com.finconsgroup.itserr.messaging.producer.ProducerRegistry;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.Map;

import static com.finconsgroup.itserr.marketplace.userprofile.bs.util.Constants.SERVICE_FILTER_KEY;
import static com.finconsgroup.itserr.marketplace.userprofile.bs.util.Constants.SERVICE_FILTER_VALUE;

/**
 * Default implementation of {@link EventProducer} that publishes
 * resource events (create, update, delete) for an event.
 */
@Component
@ConditionalOnProperty(name = "messaging.enabled", havingValue = "true")
@Slf4j
public class DefaultEventProducer implements EventProducer {

    private final MessageProducer messageProducer;

    private final UserProfileBsConfigurationProperties.Messaging messagingProperties;

    /**
     * Constructs the DefaultNotificationProducer.
     *
     * @param bsConfigurationProperties application-specific configuration properties
     * @param producerRegistry          the registry providing message producers
     */
    public DefaultEventProducer(
            UserProfileBsConfigurationProperties bsConfigurationProperties,
            ProducerRegistry producerRegistry
    ) {
        this.messagingProperties = bsConfigurationProperties.getMessaging();
        this.messageProducer = producerRegistry.getMessageProducer(this.messagingProperties.getEventProducerName());
    }

    @Override
    public void publishCreateEvent(@NonNull MessagingEventDto<?> eventDto) {
        publish(eventDto, messagingProperties.getEventCreatedType());
    }

    @Override
    public void publishUpdateEvent(@NonNull MessagingEventDto<?> eventDto) {
        publish(eventDto, messagingProperties.getEventUpdatedType());
    }

    @Override
    public void publishDeleteEvent(@NonNull MessagingEventDto<?> eventDto) {
        publish(eventDto, messagingProperties.getEventDeletedType());
    }

    @Override
    public void publishUpdateCVEvent(@NonNull MessagingEventDto<?> eventDto) {
        publish(eventDto, messagingProperties.getEventCvUpdatedType());
    }

    @Override
    public void publishUpdateImageEvent(@NonNull MessagingEventDto<?> eventDto) {
        publish(eventDto, messagingProperties.getEventImageUpdatedType());
    }

    @Override
    public void publishUpdateLanguageEvent(@NonNull MessagingEventDto<?> eventDto) {
        publish(eventDto, messagingProperties.getEventLanguageUpdatedType());
    }

    @Override
    public void publishInterestsEvent(@NonNull MessagingEventDto<?> eventDto) {
        publish(eventDto, messagingProperties.getEventInterestsType());
    }

    @Override
    public void publishCatalogInterestsEvent(@NonNull MessagingEventDto<?> eventDto) {
        publish(eventDto, messagingProperties.getEventCatalogInterestsType());
    }

    @Override
    public void publishEndorsementEvent(@NonNull MessagingEventDto<?> eventDto) {
        publish(eventDto, messagingProperties.getEventEndorsementType());
    }

    /**
     * Sends a message to the messaging system.
     *
     * @param eventDto  the payload of the message
     * @param eventType the type of event to publish
     * @throws WP2MessagingException if message publishing fails
     */
    private void publish(@NonNull MessagingEventDto<?> eventDto, @NonNull String eventType) {
        try {
            Map<String, MessagingEventDto<?>> messageMap = Map.of(eventDto.getId(), eventDto);
            Map<String, String> filterProperties = Map.of(SERVICE_FILTER_KEY, SERVICE_FILTER_VALUE);
            log.debug("Sending message, messageMap: {}, eventType: {}, source: {}, filterProperties: {}",
                    messageMap, eventType, messagingProperties.getSource(), filterProperties);
            messageProducer.publishEvent(messageMap, eventType, messagingProperties.getSource(), filterProperties);
            log.info("Successfully published event message for resource with id: {}", eventDto.getId());
        } catch (Exception e) {
            String errorMessage = "Error occurred while sending event message: %s".formatted(e.getMessage());
            throw new WP2MessagingException(errorMessage);
        }
    }
}
