package com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.impl;


import com.finconsgroup.itserr.marketplace.userprofile.bs.config.properties.UserProfileBsConfigurationProperties;
import com.finconsgroup.itserr.marketplace.userprofile.bs.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.ResourceProducer;
import com.finconsgroup.itserr.messaging.producer.MessageProducer;
import com.finconsgroup.itserr.messaging.producer.ProducerRegistry;
import com.finconsgroup.itserr.messaging.exception.WP2MessagingException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Default implementation of {@link ResourceProducer} that publishes
 * resource events (create, update, delete) for user profile.
 */
@ConditionalOnProperty(name = "messaging.enabled", havingValue = "true")
@Component
@Slf4j
public class DefaultResourceProducer implements ResourceProducer {

    private final MessageProducer messageProducer;

    private final UserProfileBsConfigurationProperties.Messaging messagingProperties;

    /**
     * Constructs the DefaultResourceProducer.
     *
     * @param userProfileBsConfigurationProperties application-specific configuration properties
     * @param producerRegistry                     the registry providing message producers
     */
    public DefaultResourceProducer(
            UserProfileBsConfigurationProperties userProfileBsConfigurationProperties,
            ProducerRegistry producerRegistry
    ) {
        this.messagingProperties = userProfileBsConfigurationProperties.getMessaging();
        this.messageProducer = producerRegistry.getMessageProducer(this.messagingProperties.getProducerName());
    }

    @Override
    public void publishCreatedResource(@NonNull OutputUserProfileDto resource) {
        publish(Map.of(resource.getId().toString(), resource), messagingProperties.getResourceCreatedType());
    }

    @Override
    public void publishUpdatedResource(@NonNull OutputUserProfileDto resource) {
        publish(Map.of(resource.getId().toString(), resource), messagingProperties.getResourceUpdatedType());
    }

    @Override
    public void publishUpdatedResources(@NonNull List<OutputUserProfileDto> resources) {
        publish(toMessageMap(resources), messagingProperties.getResourceUpdatedType());
    }

    @Override
    public void publishDeletedResource(@NonNull OutputUserProfileDto resource) {
        publish(Map.of(resource.getId().toString(), resource), messagingProperties.getResourceDeletedType());
    }

    /**
     * Publishes an event to the messaging system.
     *
     * @param messageMap the message payload
     * @param eventType  the type of event
     * @throws WP2MessagingException if message publishing fails
     */
    private void publish(@NonNull Map<String, OutputUserProfileDto> messageMap, @NonNull String eventType) {
        try {
            Map<String, String> filterProperties = Map.of(SERVICE_FILTER_KEY, SERVICE_FILTER_VALUE);
            log.debug("Sending message, messageMap: {}, eventType: {}, source: {}, filterProperties: {}",
                    messageMap, eventType, messagingProperties.getSource(), filterProperties);
            messageProducer.publishEvent(messageMap, eventType, messagingProperties.getSource(), filterProperties);
            log.info("Successfully published message for resource(s) with id(s):{}", messageMap.keySet());
        } catch (Exception e) {
            String errorMessage = "Error occurred while sending message: %s".formatted(e.getMessage());
            throw new WP2MessagingException(errorMessage);
        }
    }

    /**
     * Converts a list of DTOs into a map of id -> DTO.
     *
     * @param resources list of resources
     * @return map of id to resource
     */
    private Map<String, OutputUserProfileDto> toMessageMap(List<OutputUserProfileDto> resources) {
        return resources.stream()
                .collect(Collectors.toMap(
                        resource -> resource.getId().toString(),
                        resource -> resource
                ));
    }

}
