package com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.dto;

import com.finconsgroup.itserr.marketplace.userprofile.bs.messaging.enums.OperationType;
import com.finconsgroup.itserr.messaging.dto.UntypedMessagingEventDto;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import org.springframework.lang.NonNull;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

/**
 * A wrapper class to standardize handling of different event data types in the messaging system.
 * This class encapsulates common attributes across various notification data types, such as
 * institutional page status changes, join requests, member invitations, and membership changes,
 * ensuring uniform processing in event subscribers.
 */
@Data
@Builder
@AllArgsConstructor
public class EventDataWrapper {

    /**
     * The unique identifier of the resource associated with the event.
     */
    private final String resourceId;

    /**
     * The name of the resource associated with the event.
     */
    private final String name;

    /**
     * Indicates the type of event
     */
    private final String eventType;

    /**
     * The list of user IDs affected by the event.
     */
    private final List<UUID> userIds;

    /**
     * The type of operation associated with the event (e.g., CREATE, DELETE).
     * May be null for events that do not involve specific operations, such as join requests
     * or membership changes.
     */
    private final OperationType operationType;

    /**
     * The unique identifier of the root institutional page associated with the event.
     * May be null for certain event types, such as membership changes.
     */
    private final UUID rootInstitutionalPageId;

    /**
     * The name of the root institutional page associated with the event.
     * May be null for certain event types, such as membership changes.
     */
    private final String rootInstitutionalPageName;

    /**
     * Map of Ids and names of all associated institutional pages in the hierarchy
     * including the root institutional page
     */
    private final Map<UUID, String> hierarchyInstitutionalPages;

    /**
     * Indicates whether the event has been approved for processing.
     */
    private final Boolean approved;

    /**
     * Indicates whether the event pertains to a work package (WP) leader role.
     */
    private final Boolean wpLeader;

    public EventDataWrapper(@NonNull UntypedMessagingEventDto resourcePayload,
                            @NonNull String eventType,
                            @NonNull InstitutionalPageStatusChangeNotificationData data,
                            @NonNull List<UUID> userIds,
                            @NonNull Boolean wpLeader) {
        this.resourceId = resourcePayload.getId();
        this.name = resourcePayload.getName();
        this.eventType = eventType;
        this.userIds = userIds;
        this.operationType = data.getOperationType();
        this.rootInstitutionalPageId = data.getRootInstitutionalPageId();
        this.rootInstitutionalPageName = data.getRootInstitutionalPageName();
        this.approved = data.getApproved();
        this.wpLeader = wpLeader;
        this.hierarchyInstitutionalPages = new HashMap<>();
    }

    public EventDataWrapper(@NonNull UntypedMessagingEventDto resourcePayload,
                            @NonNull String eventType,
                            @NonNull JoinRequestStatusChangeNotificationData data,
                            @NonNull List<UUID> userIds,
                            @NonNull Boolean wpLeader) {
        this.resourceId = resourcePayload.getId();
        this.name = resourcePayload.getName();
        this.eventType = eventType;
        this.userIds = userIds;
        this.operationType = null;
        this.rootInstitutionalPageId = data.getRootInstitutionalPageId();
        this.rootInstitutionalPageName = data.getRootInstitutionalPageName();
        this.approved = data.getApproved();
        this.wpLeader = wpLeader;
        this.hierarchyInstitutionalPages = new HashMap<>();
    }

    public EventDataWrapper(@NonNull UntypedMessagingEventDto resourcePayload,
                            @NonNull String eventType,
                            @NonNull MemberInvitationStatusChangeNotificationData data,
                            @NonNull List<UUID> userIds,
                            @NonNull Boolean wpLeader) {
        this.resourceId = resourcePayload.getId();
        this.name = resourcePayload.getName();
        this.eventType = eventType;
        this.userIds = userIds;
        this.operationType = null;
        this.rootInstitutionalPageId = data.getRootInstitutionalPageId();
        this.rootInstitutionalPageName = data.getRootInstitutionalPageName();
        this.approved = data.getApproved();
        this.wpLeader = wpLeader;
        this.hierarchyInstitutionalPages = data.getHierarchyInstitutionalPages();
    }

    public EventDataWrapper(@NonNull UntypedMessagingEventDto resourcePayload,
                            @NonNull String eventType,
                            @NonNull MembershipChangeNotificationData data,
                            @NonNull List<UUID> userIds,
                            @NonNull Boolean wpLeader) {
        this.resourceId = resourcePayload.getId();
        this.name = resourcePayload.getName();
        this.eventType = eventType;
        this.userIds = userIds;
        this.operationType = null;
        this.rootInstitutionalPageId = data.getRootInstitutionalPageId();
        this.rootInstitutionalPageName = data.getRootInstitutionalPageName();
        this.approved = data.getWpLeader();
        this.wpLeader = wpLeader;
        this.hierarchyInstitutionalPages = data.getHierarchyInstitutionalPages();
    }

}