package com.finconsgroup.itserr.marketplace.userprofile.bs.api;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.InputAdminPatchUserProfileDto;
import com.finconsgroup.itserr.marketplace.userprofile.bs.client.dto.OutputAdminPatchUserProfileDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;

/**
 * Interface for managing REST operations on UserProfile.
 *
 * <p>
 * Provides endpoints for admin to update user profile
 * </p>
 */
@Tag(
        name = "AdminApi",
        description = "API for managing user profiles"
)
@SecurityRequirement(name = "BearerAuth")
public interface AdminApi {

    /**
     * Patch the user profile.
     *
     * @param profileId                     id of the user profile
     * @param inputAdminPatchUserProfileDto which contains information that needs to be updated for an existing entity
     * @return the found {@link OutputAdminPatchUserProfileDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "patch a user profile by ID",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }),
            }
    )
    @PatchMapping(value = "/admin/profile/{profileId}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputAdminPatchUserProfileDto patchUserProfile(@PathVariable("profileId") UUID profileId,
                                                    @RequestBody InputAdminPatchUserProfileDto inputAdminPatchUserProfileDto);


    /**
     * Publish resource messages for all user profiles.
     * <p></p>
     * One of the use case would be to sync all user profiles with the Search index.
     */
    @Operation(
            summary = "publish resource messages for all user profiles",
            responses = {
                    @ApiResponse(responseCode = "204", description = "No Content")
            }
    )
    @PostMapping(value = "/admin/profiles/publish")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void publishAllProfilesMessages();
}
