package com.finconsgroup.itserr.marketplace.usercommunication.dm.util;

import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;

import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

public final class SortUtils {

    /**
     * Maps the sort property name to the name that can be used to sort the entities and
     * returns the new sort.
     *
     * @param propertyMap the map containing name that should be used for queries instead of the property name
     * @param sort        the original sort
     * @return Sort with property name mapped if necessary
     */
    @NonNull
    public static Sort mapSortProperty(@Nullable Map<String, String> propertyMap, @NonNull Sort sort) {
        final AtomicReference<Sort> sortResult = new AtomicReference<>(Sort.unsorted());
        sort
                .stream().sequential()
                .map(o -> Sort.by(o.getDirection(), mapSortProperty(propertyMap, o.getProperty())))
                .forEach(newSort -> sortResult.getAndAccumulate(newSort, Sort::and));
        return sortResult.get();
    }

    /*
     * Maps the sort property name to name that can be used to sort the entities.
     * e.g. text fields cannot be used to sort directly, and instead we need to use
     * the field data of type keyword
     *
     * @param propertyMap the map containing name that should be used for queries instead of the property name
     * @param property    the property name to be mapped
     * @return the mapped property name, if found, otherwise the provided property name
     */
    @NonNull
    private static String mapSortProperty(@Nullable Map<String, String> propertyMap, @NonNull String property) {
        return Optional.ofNullable(propertyMap)
                .map(m -> m.getOrDefault(property, property))
                .orElse(property);
    }

}