package com.finconsgroup.itserr.marketplace.usercommunication.dm.service.impl;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.configuration.properties.RabbitMQMessagingClientProperties;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ChatMessage;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.MessagingService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;

import static com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessagingType.RABBIT;

/**
 * Implementation for {@link MessagingService} that sends the messages to Rabbit MQ queues.
 */
@Service
@ConditionalOnProperty(value = "type", prefix = "user-communication.dm.messaging", havingValue = RABBIT)
@RequiredArgsConstructor
@Slf4j
public class RabbitMQMessagingService implements MessagingService {

    private final RabbitTemplate rabbitTemplate;
    private final RabbitMQMessagingClientProperties rabbitMQMessagingClientProperties;

    @Override
    public void sendChatMessage(@NonNull ChatMessage chatMessage) {
        log.info("Sending chat message: {} from user: {} to exchange: {}",
                chatMessage.getId(), chatMessage.getSenderId(),
                rabbitMQMessagingClientProperties.getChat().getExchange());

        rabbitTemplate.convertAndSend(
                rabbitMQMessagingClientProperties.getChat().getExchange(),
                rabbitMQMessagingClientProperties.getChat().getChatRoutingKey(),
                chatMessage
        );
    }

    @Override
    public void sendUserNotification(@NonNull ChatMessage notification) {
        rabbitTemplate.convertAndSend(
                rabbitMQMessagingClientProperties.getChat().getExchange(),
                rabbitMQMessagingClientProperties.getChat().getUserNotificationRoutingKey(),
                notification
        );
    }
}
