package com.finconsgroup.itserr.marketplace.usercommunication.dm.service.impl;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessageDestinations;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputUnreadMessageSummaryDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ChatMessage;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.OfflineMessage;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.MessageType;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.OfflineMessageRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.ChatMessageProducer;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.OfflineMessageService;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.SessionManagementService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.messaging.simp.SimpMessagingTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessageHeaders.USER_DESTINATION_DEFAULTS;
import static com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.SecurityConstants.SYSTEM_USER_ID;

/**
 * Service implementation handling offline message queuing, notifications, and delivery.
 */
@Service
@Transactional
@RequiredArgsConstructor
@Slf4j
public class OfflineMessageServiceImpl implements OfflineMessageService {


    private final OfflineMessageRepository offlineMessageRepository;
    private final SimpMessagingTemplate messagingTemplate;
    private final ChatMessageProducer messageProducer;
    private final SessionManagementService sessionManagementService;
    private final MessageDestinations messageDestinations;

    @Override
    public void queueMessageForOfflineUser(ChatMessage chatMessage, UUID recipientUserId) {
        try {
            if (offlineMessageRepository.existsByMessageIdAndRecipientUserId(
                    chatMessage.getId(), recipientUserId)) {
                log.debug("Message {} already queued for user {}",
                           chatMessage.getId(), recipientUserId);
                return;
            }
            OfflineMessage offlineMessage = new OfflineMessage(
                chatMessage.getId(),
                recipientUserId,
                chatMessage.getConversationId(),
                chatMessage.getSenderId(),
                chatMessage.getContent(),
                chatMessage.getMessageType(),
                chatMessage.getCreatedAt()
            );
            offlineMessageRepository.save(offlineMessage);
            log.info("Queued message {} for offline user {}",
                       chatMessage.getId(), recipientUserId);
        } catch (Exception e) {
            log.error("Failed to queue message {} for user {}: {}",
                        chatMessage.getId(), recipientUserId, e.getMessage(), e);
        }
    }

    @Override
    public void notifyUserAboutPendingMessages(UUID userId) {
        try {
            long messageCount = getUndeliveredMessageCount(userId);
            if (messageCount > 0) {
                List<OfflineMessage> recentMessages = getUndeliveredMessages(userId);
                String lastSenderName = null;
                String lastMessageContent = null;
                if (!recentMessages.isEmpty()) {
                    OfflineMessage lastMessage = recentMessages.get(recentMessages.size() - 1);
                    lastMessageContent = lastMessage.getContent();
                }
                String notificationContent;
                if (lastSenderName != null) {
                    if (messageCount == 1) {
                        notificationContent = String.format("%s sent you a message. Connect to read it.", lastSenderName);
                        if (lastMessageContent != null && !lastMessageContent.isEmpty()) {
                            String truncatedContent = lastMessageContent.length() > 50 
                                ? lastMessageContent.substring(0, 50) + "..." 
                                : lastMessageContent;
                            notificationContent = String.format("%s sent you a message: \"%s\". Connect to read the full message.", 
                                lastSenderName, truncatedContent);
                        }
                    } else {
                        notificationContent = String.format("You have %d unread messages waiting for you. Latest from %s. Connect to read them.", 
                            messageCount, lastSenderName);
                        if (lastMessageContent != null && !lastMessageContent.isEmpty()) {
                            String truncatedContent = lastMessageContent.length() > 50 
                                ? lastMessageContent.substring(0, 50) + "..." 
                                : lastMessageContent;
                            notificationContent = String.format("You have %d unread messages waiting for you. Latest from %s: \"%s\". Connect to read them.", 
                                messageCount, lastSenderName, truncatedContent);
                        }
                    }
                } else {
                    notificationContent = String.format("You have %d unread message%s waiting for you. Connect to read %s.", 
                        messageCount, messageCount == 1 ? "" : "s", messageCount == 1 ? "it" : "them");
                }
                messageProducer.sendUserNotification(userId, notificationContent, MessageType.NOTIFICATION);
                ChatMessage notification = new ChatMessage();
                notification.setId(java.util.UUID.randomUUID());
                notification.setSenderId(SYSTEM_USER_ID);
                notification.setReceiverId(userId);
                notification.setContent(notificationContent);
                notification.setMessageType(MessageType.NOTIFICATION);
                notification.setCreatedAt(Instant.now());
                if (lastSenderName != null) {
                    String enhancedContent = notification.getContent() + 
                        "||SENDER:" + lastSenderName + 
                        (lastMessageContent != null ? "||MESSAGE:" + lastMessageContent : "");
                    notification.setContent(enhancedContent);
                }
                messagingTemplate.convertAndSendToUser(userId.toString(), messageDestinations.getUserNotificationsSuffix(),
                        notification, USER_DESTINATION_DEFAULTS);
                log.info("Sent pending messages notification to user {}: {} messages from {}", 
                           userId, messageCount, lastSenderName != null ? lastSenderName : "unknown");
            }
        } catch (Exception e) {
            log.error("Failed to send pending messages notification to user {}: {}", userId, e.getMessage(), e);
        }
    }

    @Override
    public void deliverOfflineMessages(UUID userId) {
        try {
            List<OfflineMessage> queuedMessages = offlineMessageRepository
                .findByRecipientUserIdAndDeliveredFalseOrderByCreatedAtAsc(userId);
            if (queuedMessages.isEmpty()) {
                log.debug("No offline messages to deliver for user {}", userId);
                return;
            }
            notifyUserAboutPendingMessages(userId);
            log.info("Delivering {} offline messages to user {}", queuedMessages.size(), userId);
            for (OfflineMessage offlineMessage : queuedMessages) {
                try {
                    ChatMessage chatMessage = convertToCharMessage(offlineMessage);
                    chatMessage.setContent(chatMessage.getContent() + "||BULK_OFFLINE_DELIVERY||");
                    messagingTemplate.convertAndSendToUser(userId.toString(), messageDestinations.getUserMessagesSuffix(),
                            chatMessage, USER_DESTINATION_DEFAULTS);
                    messagingTemplate.convertAndSend(messageDestinations.getConversationTopic(offlineMessage.getConversationId()), chatMessage);
                    offlineMessage.setDelivered(true);
                    offlineMessageRepository.save(offlineMessage);
                    log.debug("Delivered offline message {} to user {}", 
                               offlineMessage.getMessageId(), userId);
                } catch (Exception e) {
                    log.error("Failed to deliver offline message {} to user {}: {}", 
                               offlineMessage.getMessageId(), userId, e.getMessage(), e);
                }
            }
            log.info("Successfully delivered {} offline messages to user {}", queuedMessages.size(), userId);
        } catch (Exception e) {
            log.error("Failed to deliver offline messages to user {}: {}", userId, e.getMessage(), e);
        }
    }

    @Override
    public boolean isUserOnline(UUID userId) {
        return sessionManagementService.isUserOnline(userId);
    }

    @Override
    public long getUndeliveredMessageCount(UUID userId) {
        return offlineMessageRepository.countByRecipientUserIdAndDeliveredFalse(userId);
    }

    @Override
    public long getUndeliveredMessageCount(UUID userId, UUID conversationId) {
        return offlineMessageRepository.countByRecipientUserIdAndConversationIdAndDeliveredFalse(userId, conversationId);
    }

    @Override
    public List<OfflineMessage> getUndeliveredMessages(UUID userId) {
        return offlineMessageRepository.findByRecipientUserIdAndDeliveredFalseOrderByCreatedAtAsc(userId);
    }

    @Override
    public OutputUnreadMessageSummaryDto getUnreadMessageSummary(UUID userId) {
        // Get total unread message count
        long totalUnreadCount = getUndeliveredMessageCount(userId);

        if (totalUnreadCount == 0) {
            return new OutputUnreadMessageSummaryDto(0, null);
        }

        // Get recent unread messages with sender information
        Optional<OfflineMessage> earliestUnreadMessage = offlineMessageRepository.findTop1ByRecipientUserIdAndDeliveredFalseOrderByCreatedAtAsc(userId);

        // Extract sender information from the most recent message
        String lastSenderMessage = earliestUnreadMessage.map(OfflineMessage::getContent).orElse(null);

        return new OutputUnreadMessageSummaryDto(
                totalUnreadCount,
                lastSenderMessage
        );
    }

    @Override
    public void cleanupOldDeliveredMessages(int daysOld) {
        try {
            Instant cutoffDate = Instant.now().minus(daysOld, ChronoUnit.DAYS);
            int deletedCount = offlineMessageRepository.deleteDeliveredMessagesOlderThan(cutoffDate);
            if (deletedCount > 0) {
                log.info("Cleaned up {} old delivered messages older than {} days", 
                           deletedCount, daysOld);
            }
        } catch (Exception e) {
            log.error("Failed to cleanup old delivered messages: {}", e.getMessage(), e);
        }
    }

    @Override
    public void handleMessageDelivery(ChatMessage chatMessage, UUID recipientUserId) {
        if (isUserOnline(recipientUserId)) {
            try {
                messagingTemplate.convertAndSendToUser(recipientUserId.toString(),
                        messageDestinations.getUserMessagesSuffix(), chatMessage, USER_DESTINATION_DEFAULTS);
                chatMessage.setDelivered(true);
                log.debug("Delivered message {} directly to online user {}", 
                           chatMessage.getId(), recipientUserId);
            } catch (Exception e) {
                log.error("Failed to deliver message {} to online user {}: {}", 
                           chatMessage.getId(), recipientUserId, e.getMessage(), e);
                queueMessageForOfflineUser(chatMessage, recipientUserId);
            }
        } else {
            queueMessageForOfflineUser(chatMessage, recipientUserId);
            log.info("Queued message for offline user {}. They will be notified when they come online.", recipientUserId);
        }
    }

    @Override
    public void sendOfflineMessageNotification(UUID userId) {
        try {
            long messageCount = getUndeliveredMessageCount(userId);
            if (messageCount > 0) {
                ChatMessage notification = new ChatMessage();
                notification.setId(UUID.randomUUID());
                notification.setSenderId(SYSTEM_USER_ID);
                notification.setReceiverId(userId);
                notification.setMessageType(MessageType.SYSTEM);
                notification.setCreatedAt(Instant.now());
                String notificationText = messageCount == 1
                        ? "You have 1 new message waiting for you."
                        : String.format("You have %d new messages waiting for you.", messageCount);
                notification.setContent(notificationText);
                messagingTemplate.convertAndSendToUser(userId.toString(), messageDestinations.getUserNotificationsSuffix(),
                        notification, USER_DESTINATION_DEFAULTS);
                log.info("Sent offline message notification to user {}: {} messages pending", userId, messageCount);
            }
        } catch (Exception e) {
            log.error("Failed to send offline message notification to user {}: {}", userId, e.getMessage(), e);
        }
    }

    private ChatMessage convertToCharMessage(OfflineMessage offlineMessage) {
        ChatMessage chatMessage = new ChatMessage();
        chatMessage.setId(offlineMessage.getMessageId());
        chatMessage.setSenderId(offlineMessage.getSenderId());
        chatMessage.setReceiverId(offlineMessage.getRecipientUserId());
        chatMessage.setContent(offlineMessage.getContent());
        chatMessage.setMessageType(offlineMessage.getMessageType());
        chatMessage.setCreatedAt(offlineMessage.getCreatedAt());
        chatMessage.setConversationId(offlineMessage.getConversationId());
        chatMessage.setDelivered(true);
        chatMessage.setDeliveredAt(Instant.now());
        return chatMessage;
    }
}
