package com.finconsgroup.itserr.marketplace.usercommunication.dm.service.impl;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputChatMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputTypingMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputChatMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ChatMessage;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.Conversation;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.MessageType;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.mapper.ChatMessageMapper;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.ChatMessageProducer;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.ChatMessageService;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.ConversationService;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.DirectConversationService;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.OfflineMessageService;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.PreConditionService;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.WebSocketService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.time.Instant;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.SecurityConstants.SYSTEM_USER_ID;

/**
 * Default implementation for {@link WebSocketService}
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultWebSocketService implements WebSocketService {

    private final PreConditionService preConditionService;
    private final ChatMessageProducer messageProducer;
    private final ConversationService conversationService;
    private final DirectConversationService directConversationService;
    private final OfflineMessageService offlineMessageService;
    private final ChatMessageMapper chatMessageMapper;
    private final ChatMessageService chatMessageService;

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public OutputChatMessageDto sendMessage(UUID userId, UUID conversationId, InputChatMessageDto inputChatMessageDto) {
        Conversation conversation = preConditionService.checkUserAndConversationAndMessage(userId, conversationId,
                inputChatMessageDto.getMessageType());

        ChatMessage chatMessage = chatMessageMapper.inputChatMessageDtoToEntity(inputChatMessageDto, userId);

        // Set message metadata
        if (chatMessage.getId() == null) {
            chatMessage.setId(UUID.randomUUID());
        }
        chatMessage.setConversationId(conversationId);
        chatMessage.setCreatedAt(Instant.now());

        // For direct conversations, set the receiver ID and handle offline messages
        UUID receiverId = null;

        if (conversation.isDirect()) {
            receiverId = conversation.getOtherParticipantId(userId);
            chatMessage.setReceiverId(receiverId);
        }

        // Handle offline message delivery for direct messages
        // This works even if users haven't connected before or aren't participants yet
        if (receiverId != null) {
            offlineMessageService.handleMessageDelivery(chatMessage, receiverId);
            log.debug("Handled offline message delivery to user {} regardless of participant status", receiverId);
        }

        // Send message to Messaging Service for processing and persistence
        messageProducer.sendChatMessage(chatMessage);

        // Mark message as delivered since it's being sent via WebSocket
        chatMessage.setDelivered(true);

        log.info("WebSocket message sent to conversation {}: {}", conversationId, chatMessage.getId());
        return chatMessageService.getOutputChatMessageDto(chatMessage, false);

    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public OutputChatMessageDto sendDirectMessage(UUID userId, UUID receiverId, InputChatMessageDto inputChatMessageDto) {
        // Validate required fields
        if (userId == null || receiverId == null) {
            log.warn("Direct message missing sender or receiver ID");
            return null;
        }

        ChatMessage chatMessage = chatMessageMapper.inputChatMessageDtoToEntity(inputChatMessageDto, userId);
        chatMessage.setReceiverId(receiverId);
        // Create or get existing direct conversation
        OutputConversationDto conversationDto = directConversationService.createOrGetDirectConversation(userId, receiverId);

        chatMessage.setId(UUID.randomUUID());
        chatMessage.setConversationId(conversationDto.getId());
        chatMessage.setCreatedAt(Instant.now());

        // Send message to Messaging Service for processing and persistence
        messageProducer.sendChatMessage(chatMessage);

        return chatMessageService.getOutputChatMessageDto(chatMessage, false);
    }

    @Override
    public OutputChatMessageDto sendTypingIndicator(UUID userId, UUID conversationId, InputTypingMessageDto inputTypingMessageDto) {

        preConditionService.checkUserAndConversation(userId, conversationId);

        ChatMessage chatMessage = new ChatMessage();
        chatMessage.setConversationId(conversationId);
        chatMessage.setSenderId(userId);
        chatMessage.setContent(inputTypingMessageDto.getContent());
        chatMessage.setMessageType(MessageType.TYPING);
        chatMessage.setCreatedAt(Instant.now());

        log.debug("Typing indicator sent for user {} in conversation {}", userId, conversationId);
        return chatMessageService.getOutputChatMessageDto(chatMessage, false);
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public OutputChatMessageDto addUser(UUID userId, UUID conversationId, InputChatMessageDto inputChatMessageDto) {

        // Validate conversation exists and user can join
        Conversation conversation = preConditionService.checkConversation(conversationId);

        ChatMessage chatMessage = chatMessageMapper.inputChatMessageDtoToEntity(inputChatMessageDto, userId);

        // For direct conversations, users are automatically participants
        // For groups, check if user can join
        if (!conversation.isDirect()) {
            if (conversation.isGroup() && conversation.isPrivateFlag()) {
                // Private groups require invitation
                if (!conversationService.isUserParticipant(conversationId, userId)) {
                    log.warn("User {} attempted to join private group {} without invitation",
                            userId, conversationId);
                    return null;
                }
            }
        }

        // Create join message
        chatMessage.setId(UUID.randomUUID());
        chatMessage.setMessageType(MessageType.JOIN);
        chatMessage.setConversationId(conversationId);
        chatMessage.setContent(userId + " joined the conversation");
        chatMessage.setCreatedAt(Instant.now());

        // Send join notification through messaging service
        messageProducer.sendChatMessage(chatMessage);

        log.info("User {} joined conversation {} via WebSocket", userId, conversationId);
        return chatMessageService.getOutputChatMessageDto(chatMessage, false);
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public OutputChatMessageDto removeUser(UUID userId, UUID conversationId, InputChatMessageDto inputChatMessageDto) {

        // Validate user is participant
        preConditionService.checkUserAndConversation(userId, conversationId);

        ChatMessage chatMessage = chatMessageMapper.inputChatMessageDtoToEntity(inputChatMessageDto, userId);
        // Leave the conversation
        conversationService.leaveConversation(conversationId, userId);

        // Create leave message
        chatMessage.setId(UUID.randomUUID());
        chatMessage.setMessageType(MessageType.LEAVE);
        chatMessage.setConversationId(conversationId);
        chatMessage.setContent(userId + " left the conversation");
        chatMessage.setCreatedAt(Instant.now());

        // Send leave notification through Messaging Service
        messageProducer.sendChatMessage(chatMessage);

        log.info("User {} left conversation {} via WebSocket", userId, conversationId);
        return chatMessageService.getOutputChatMessageDto(chatMessage, false);
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public OutputChatMessageDto broadcastSystemMessage(UUID conversationId, String content) {
        ChatMessage systemMessage = new ChatMessage();
        systemMessage.setId(UUID.randomUUID());
        systemMessage.setSenderId(SYSTEM_USER_ID);
        systemMessage.setContent(content);
        systemMessage.setMessageType(MessageType.SYSTEM);
        systemMessage.setConversationId(conversationId);
        systemMessage.setCreatedAt(Instant.now());

        // Send to Messaging Service for persistence
        messageProducer.sendChatMessage(systemMessage);

        return chatMessageService.getOutputChatMessageDto(systemMessage, false);
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public OutputChatMessageDto sendUserNotification(UUID userId, String content, MessageType messageType) {
        ChatMessage notification = new ChatMessage();
        notification.setId(UUID.randomUUID());
        notification.setSenderId(SYSTEM_USER_ID);
        notification.setReceiverId(userId);
        notification.setContent(content);
        notification.setMessageType(messageType);
        notification.setCreatedAt(Instant.now());

        // Send to Messaging Service for persistence
        messageProducer.sendUserNotification(userId, content, messageType);

        return chatMessageService.getOutputChatMessageDto(notification, false);
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public OutputChatMessageDto sendInvitation(UUID userId, UUID conversationId, UUID receiverId) {

        Conversation conversation = preConditionService.checkConversation(conversationId);

        String content = String.format("You have been invited to join '%s' by %s",
                conversation.getName(), userId);

        ChatMessage inviteNotification = new ChatMessage();
        inviteNotification.setId(UUID.randomUUID());
        inviteNotification.setSenderId(userId);
        inviteNotification.setReceiverId(receiverId);
        inviteNotification.setContent(content);
        inviteNotification.setMessageType(MessageType.NOTIFICATION);
        inviteNotification.setConversationId(conversationId);
        inviteNotification.setCreatedAt(Instant.now());

        // Send to Messaging Service for persistence
        messageProducer.sendUserNotification(receiverId, content, MessageType.NOTIFICATION);

        return chatMessageService.getOutputChatMessageDto(inviteNotification, false);
    }
}
