package com.finconsgroup.itserr.marketplace.usercommunication.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ValidationException;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.component.ConversationHelper;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputBulkMessagesDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputChatMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationMessageSummaryDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationParticipantDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationStatsDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputSearchResultDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ChatMessage;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.Conversation;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ConversationParticipant;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.ConversationType;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.mapper.ConversationMapper;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.mapper.ConversationParticipantMapper;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.ChatMessageRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.ConversationParticipantRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.ConversationRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.MessageReadReceiptRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.OfflineMessageRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.ChatMessageService;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.ConversationService;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.PreConditionService;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.util.SortUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import static com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.ConversationRepository.FIELD_COLUMN_NAME_MAP;

/**
 * Service implementation for conversation management (direct and group).
 * Contains business rules for participants, permissions, read receipts, and stats.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ConversationServiceImpl implements ConversationService {

    private final ConversationRepository conversationRepository;
    private final ConversationParticipantRepository participantRepository;
    private final ChatMessageService chatMessageService;
    private final ChatMessageRepository chatMessageRepository;
    private final MessageReadReceiptRepository readReceiptRepository;
    private final OfflineMessageRepository offlineMessageRepository;
    private final ConversationMapper conversationMapper;
    private final PreConditionService preConditionService;
    private final ConversationHelper conversationHelper;
    private final ConversationParticipantMapper conversationParticipantMapper;

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public OutputConversationDto getConversation(UUID userId, UUID conversationId) {
        Conversation conversation = preConditionService.checkUserAndConversation(userId,
                conversationId);
        OutputConversationDto conversationDto;
        if (conversation.isDirect()) {
            conversationDto = mapToDirectOutputConversationDto(userId, conversation);
        } else {
            conversationDto = conversationMapper.entityToOutputConversationDto(conversation);
        }

        Map<UUID, OutputConversationMessageSummaryDto> messageSummaryByConversationId =
                chatMessageService.getConversationMessageSummary(userId, List.of(conversation));
        conversationDto.setMessageSummary(messageSummaryByConversationId.get(conversationId));
        return conversationDto;
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public Page<OutputConversationDto> getRecentConversationsForUser(UUID userId, Pageable pageable) {
        Page<Conversation> recentConversationsPage = conversationRepository.findConversationsForUser(userId, pageable);

        Page<OutputConversationDto> conversationDtoPage = mapToOutputConverstationDtoPage(userId, recentConversationsPage);
        Map<UUID, OutputConversationMessageSummaryDto> messageSummaryByConversationId =
                chatMessageService.getConversationMessageSummary(userId, recentConversationsPage.getContent());
        conversationDtoPage.getContent().forEach(conversationDto ->
                conversationDto.setMessageSummary(messageSummaryByConversationId.get(conversationDto.getId()))
        );

        return conversationDtoPage;
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public Page<OutputChatMessageDto> getConversationMessages(UUID userId, UUID conversationId, Pageable pageable) {
        Conversation conversation = preConditionService.checkUserAndConversation(userId,
                conversationId);
        return chatMessageService.getConversationMessages(conversation, pageable);
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public List<OutputConversationParticipantDto> getConversationParticipants(UUID conversationId) {
        List<ConversationParticipant> participants = participantRepository.findByConversationIdAndActiveIsTrue(conversationId);
        return participants.stream().map(conversationParticipantMapper::entityToOutputConversationParticipantDto).toList();
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public boolean isUserParticipant(UUID conversationId, UUID userId) {
        return participantRepository.existsByConversationIdAndUserIdAndActiveIsTrue(conversationId, userId);
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public void muteConversation(UUID conversationId, UUID userId, boolean muted) {
        if (!participantRepository.existsByConversationIdAndUserIdAndActiveIsTrue(conversationId, userId)) {
            throw new WP2ValidationException("User it not an active participant");
        }
        participantRepository.updateParticipantMuteStatus(conversationId, userId, muted);
        log.info("User {} {} conversation {}", userId, muted ? "muted" : "unmuted", conversationId);
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public void markConversationAsRead(@NonNull UUID userId, @NonNull UUID conversationId) {
        Objects.requireNonNull(userId, "userId must not be null");
        Objects.requireNonNull(conversationId, "conversationId must not be null");

        Conversation conversation = preConditionService.checkUserAndConversation(userId, conversationId);
        Instant readAt = Instant.now();
        participantRepository.updateLastReadAt(conversationId, userId, readAt);
        int markedCount = chatMessageService.markConversationAsRead(conversation, userId, readAt);
        log.info("Marked {} messages as read for user {} in conversation {} (type: {})",
                   markedCount, userId, conversationId, conversation.getConversationType());
        try {
            int deletedDeliveredMessages = offlineMessageRepository.deleteByRecipientUserIdAndConversationIdAndDeliveredTrue(userId, conversationId);
            int deletedUndeliveredMessages = offlineMessageRepository.deleteByRecipientUserIdAndConversationIdAndDeliveredFalse(userId, conversationId);
            int totalDeleted = deletedDeliveredMessages + deletedUndeliveredMessages;
            if (totalDeleted > 0) {
                log.info("Cleaned up {} offline messages (delivered: {}, undelivered: {}) for user {} in conversation {}",
                           totalDeleted, deletedDeliveredMessages, deletedUndeliveredMessages, userId, conversationId);
            }
        } catch (Exception e) {
            log.warn("Failed to clean up offline messages for user {} in conversation {}: {}",
                       userId, conversationId, e.getMessage());
        }
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public void markConversationUptoMessageAsRead(@NonNull UUID userId,
                                                  @NonNull UUID conversationId,
                                                  @NonNull UUID messageId) {
        Objects.requireNonNull(userId, "userId must not be null");
        Objects.requireNonNull(conversationId, "conversationId must not be null");
        Objects.requireNonNull(messageId, "messageId must not be null");

        Conversation conversation = preConditionService.checkUserAndConversation(userId, conversationId);
        Instant readAt = Instant.now();
        ChatMessage chatMessage = chatMessageRepository.findById(messageId)
                .orElseThrow(() -> new WP2ResourceNotFoundException(messageId));

        if (!conversationId.equals(chatMessage.getConversationId())) {
            throw new WP2ValidationException("Message does not belong to conversation");
        }

        Instant messageCreatedAt = chatMessage.getCreatedAt();

        participantRepository.updateLastReadAt(conversationId, userId, readAt);
        int markedCount = chatMessageService.markConversationMessagesUptoCreatedAtAsRead(
                conversation, userId, readAt, messageCreatedAt
        );
        log.info("Marked {} messages as read for user {} in conversation {} (type: {})",
                markedCount, userId, conversationId, conversation.getConversationType());
        try {
            int deletedDeliveredMessages =
                    offlineMessageRepository.deleteByRecipientUserIdAndConversationIdAndDeliveredTrue(
                            userId, conversationId);
            int deletedUndeliveredMessages =
                    offlineMessageRepository.deleteByRecipientUserIdAndConversationIdAndDeliveredFalseUptoCreatedAt(
                            userId, conversationId, messageCreatedAt);
            int totalDeleted = deletedDeliveredMessages + deletedUndeliveredMessages;
            if (totalDeleted > 0) {
                log.info("Cleaned up {} offline messages (delivered: {}, undelivered: {}) for user {} in conversation {}",
                        totalDeleted, deletedDeliveredMessages, deletedUndeliveredMessages, userId, conversationId);
            }
        } catch (Exception e) {
            log.warn("Failed to clean up offline messages for user {} in conversation {}: {}",
                    userId, conversationId, e.getMessage());
        }
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public void markBulkMessagesAsRead(@NonNull UUID userId,
                                       @NonNull UUID conversationId,
                                       @NonNull InputBulkMessagesDto request) {
        Objects.requireNonNull(userId, "userId must not be null");
        Objects.requireNonNull(conversationId, "conversationId must not be null");
        Objects.requireNonNull(request, "request must not be null");

        // no messages to mark as read
        if (request.getIds() == null || request.getIds().isEmpty()) {
            throw new WP2ValidationException("Ids must not be null or empty");
        }

        Conversation conversation = preConditionService.checkUserAndConversation(userId, conversationId);
        Instant readAt = Instant.now();
        List<ChatMessage> chatMessages = chatMessageRepository.findByIdInAndConversationId(request.getIds(), conversationId);
        Set<UUID> chatMessageIds = chatMessages.stream().map(ChatMessage::getId).collect(Collectors.toSet());

        if (chatMessages.size() != request.getIds().size()) {
            if (log.isDebugEnabled()) {
                Set<UUID> missingMessageIds = new HashSet<>(request.getIds());
                missingMessageIds.removeAll(chatMessageIds);
                log.debug("The following message ids do not belong to the conversation - {}: {}", conversationId, missingMessageIds);
            }
            throw new WP2ValidationException("One or more messages do not belong to conversation");
        }

        participantRepository.updateLastReadAt(conversationId, userId, readAt);
        int markedCount = chatMessageService.markConversationBulkMessagesAsRead(conversation, userId, readAt, chatMessageIds);
        log.info("Marked {} messages as read for user {} in conversation {} (type: {})",
                markedCount, userId, conversationId, conversation.getConversationType());
        try {
            int deletedDeliveredMessages =
                    offlineMessageRepository.deleteByRecipientUserIdAndConversationIdAndDeliveredTrue(
                            userId, conversationId);
            int deletedUndeliveredMessages =
                    offlineMessageRepository.deleteByRecipientUserIdAndConversationIdAndDeliveredFalseAndMessageIdIn(
                            userId, conversationId, chatMessageIds);
            int totalDeleted = deletedDeliveredMessages + deletedUndeliveredMessages;
            if (totalDeleted > 0) {
                log.info("Cleaned up {} offline messages (delivered: {}, undelivered: {}) for user {} in conversation {}",
                        totalDeleted, deletedDeliveredMessages, deletedUndeliveredMessages, userId, conversationId);
            }
        } catch (Exception e) {
            log.warn("Failed to clean up offline messages for user {} in conversation {}: {}",
                    userId, conversationId, e.getMessage());
        }
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public List<ConversationParticipant> getConversationsWithUnreadMessages(UUID userId) {
        return participantRepository.findConversationsWithUnreadMessages(userId);
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public void leaveConversation(UUID conversationId, UUID userId) {
        Optional<ConversationParticipant> participantOpt = participantRepository.findByConversationIdAndUserId(conversationId, userId);
        if (participantOpt.isEmpty() || !participantOpt.get().isActive()) {
            throw new WP2ValidationException("User is not an active participant");
        }

        Optional<Conversation> conversationOpt = conversationRepository.findById(conversationId);
        if (conversationOpt.isPresent() && conversationOpt.get().isDirect()) {
            conversationRepository.deactivateConversation(conversationId);
            log.info("Direct conversation {} deactivated when user {} left", conversationId, userId);
        } else {
            participantRepository.removeParticipant(conversationId, userId);
            log.info("User {} left conversation {}", userId, conversationId);
        }
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public OutputSearchResultDto searchConversations(UUID userId, String searchTerm, Pageable pageable) {
        // map sort fields to column names, as the repository method uses native query
        PageRequest pageRequest = PageRequest.of(pageable.getPageNumber(), pageable.getPageSize(),
                SortUtils.mapSortProperty(FIELD_COLUMN_NAME_MAP, pageable.getSort()));
        Page<Conversation> searchResultPage = conversationRepository.findByNameContainingIgnoreCase(userId,
                searchTerm, pageRequest);
        Page<OutputConversationDto> conversationDtoPage = mapToOutputConverstationDtoPage(userId, searchResultPage);

        Page<OutputChatMessageDto> chatMessageDtoPage = chatMessageService.searchMessages(userId, searchTerm, pageable);

        return OutputSearchResultDto.builder()
                .conversations(OutputPageDto.fromPage(conversationDtoPage))
                .messages(OutputPageDto.fromPage(chatMessageDtoPage))
                .build();
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public OutputConversationStatsDto getConversationStats() {
        long directCount = conversationRepository.countByConversationTypeAndActiveTrue(ConversationType.DIRECT);
        long groupCount = conversationRepository.countByConversationTypeAndActiveTrue(ConversationType.GROUP);
        return new OutputConversationStatsDto(directCount, groupCount);
    }

    private OutputConversationDto mapToDirectOutputConversationDto(UUID userId, Conversation conversation) {
        return conversationHelper.populateDetailsForDirect(userId, conversation);
    }

    private Page<OutputConversationDto> mapToOutputConverstationDtoPage(UUID userId, Page<Conversation> conversationPage) {
        List<OutputConversationDto> conversationDtos = conversationHelper.populateDetailsForDirect(userId, conversationPage.getContent());
        return new PageImpl<>(conversationDtos, conversationPage.getPageable(), conversationPage.getTotalElements());
    }
}
