package com.finconsgroup.itserr.marketplace.usercommunication.dm.service;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputChatMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationMessageSummaryDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ChatMessage;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.Conversation;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

public interface ChatMessageService {

    /**
     * Handles chat messages from the chat queue
     */
    void handleChatMessage(ChatMessage chatMessage);

    /**
     * Handles user notification messages from queue
     */
    void handleUserNotification(ChatMessage chatMessage);

    /**
     * Get messages for a specific conversation
     */
    List<ChatMessage> getConversationMessages(UUID conversationId);

    /**
     * Get message summary for each conversation for the conversations
     */
    Map<UUID, OutputConversationMessageSummaryDto> getConversationMessageSummary(UUID userId, List<Conversation> conversations);

    /**
     * Get a page of chat messages for the conversation
     */
    Page<OutputChatMessageDto> getConversationMessages(Conversation conversation, Pageable pageable);

    /**
     * Mark all unread messages in a conversation as read by a specific user.
     */
    int markConversationAsRead(Conversation conversation, UUID userId, Instant readAt);

    /**
     * Mark all unread messages in a conversation as read by a specific user for the messages created at or before.
     */
    int markConversationMessagesUptoCreatedAtAsRead(Conversation conversation,
                                                    UUID userId,
                                                    Instant readAt,
                                                    Instant uptoCreatedAt);

    /**
     * Mark all unread messages in a conversation as read by a specific user for the passed ids.
     */
    int markConversationBulkMessagesAsRead(Conversation conversation, UUID userId, Instant readAt, Set<UUID> messageIds);

    /**
     * Get notifications for a specific user (messages where receiverId matches userId)
     */
    List<ChatMessage> getUserNotifications(UUID userId);

    /**
     * Clear notifications for a user (delete notification messages for the user)
     */
    void clearUserNotifications(UUID userId);

    /**
     * Get a page of chat messages matching the search term
     *
     * @param userId     the user ID
     * @param searchTerm free-text term
     * @param pageable   the pageable containing requested paging and sorting information
     * @return list of matching messages
     */
    Page<OutputChatMessageDto> searchMessages(UUID userId, String searchTerm, Pageable pageable);

    /**
     * Maps the chat message entity to output dto
     *
     * @param chatMessage the chat message entity
     * @param loadUserProfiles flag to indicate if user profiles should be loaded to fetch chat user details e.g. sender
     * @return the output chat message dto
     */
    OutputChatMessageDto getOutputChatMessageDto(@NonNull ChatMessage chatMessage,
                                                 boolean loadUserProfiles);
}
