package com.finconsgroup.itserr.marketplace.usercommunication.dm.service;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ChatMessage;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.MessageType;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.time.Instant;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.SecurityConstants.SYSTEM_USER_ID;

/**
 * Producer component responsible for sending chat and notification messages to Messaging Service.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ChatMessageProducer {


    private final MessagingService messagingService;

    /**
     * Sends a chat message to the Messaging Service
     */
    public void sendChatMessage(ChatMessage message) {
        try {
            // Ensure message has an ID and timestamp
            if (message.getId() == null) {
                message.setId(UUID.randomUUID());
            }
            if (message.getCreatedAt() == null) {
                message.setCreatedAt(Instant.now());
            }

            messagingService.sendChatMessage(message);

            log.info("Chat message sent successfully: {}", message.getId());

        } catch (Exception e) {
            log.error("Failed to send chat message: {}", message.getId(), e);
            throw new RuntimeException("Failed to send chat message", e);
        }
    }

    /**
     * Sends a user notification message
     */
    public void sendUserNotification(UUID userId, String content, MessageType messageType) {
        try {
            ChatMessage notification = new ChatMessage();
            notification.setId(UUID.randomUUID());
            notification.setSenderId(SYSTEM_USER_ID);
            notification.setReceiverId(userId);
            notification.setContent(content);
            notification.setMessageType(messageType);
            notification.setCreatedAt(Instant.now());

            log.info("Sending user notification to user: {} with type: {}", userId, messageType);

            messagingService.sendUserNotification(notification);

            log.info("User notification sent successfully to user: {}", userId);

        } catch (Exception e) {
            log.error("Failed to send user notification to user: {}", userId, e);
            throw new RuntimeException("Failed to send user notification", e);
        }
    }

//    /**
//     * Sends a join notification when a user joins a chat room
//     */
//    public void sendJoinNotification(UUID userId, String username, UUID chatRoomId) {
//        ChatMessage joinMessage = new ChatMessage();
//        joinMessage.setId(UUID.randomUUID());
//        joinMessage.setSenderId(userId);
//        joinMessage.setContent(username + " joined the chat");
//        joinMessage.setMessageType(MessageType.JOIN);
//        joinMessage.setChatRoomId(chatRoomId);
//        joinMessage.setCreatedAt(LocalDateTime.now());
//
//        sendChatMessage(joinMessage);
//    }
//
//    /**
//     * Sends a leave notification when a user leaves a chat room
//     */
//    public void sendLeaveNotification(UUID userId, String username, UUID chatRoomId) {
//        ChatMessage leaveMessage = new ChatMessage();
//        leaveMessage.setMessageId(UUID.randomUUID().toString());
//        leaveMessage.setSenderId(userId);
//        leaveMessage.setSenderName(username);
//        leaveMessage.setContent(username + " left the chat");
//        leaveMessage.setMessageType(MessageType.LEAVE);
//        leaveMessage.setChatRoomId(chatRoomId);
//        leaveMessage.setCreatedAt(LocalDateTime.now());
//
//        sendChatMessage(leaveMessage);
//    }
//
//    /**
//     * Sends a typing indicator message
//     */
//    public void sendTypingIndicator(UUID userId, String username, UUID chatRoomId, boolean isTyping) {
//        ChatMessage typingMessage = new ChatMessage();
//        typingMessage.setMessageId(UUID.randomUUID().toString());
//        typingMessage.setSenderId(userId);
//        typingMessage.setSenderName(username);
//        typingMessage.setContent(isTyping ? "is typing..." : "stopped typing");
//        typingMessage.setMessageType(MessageType.TYPING);
//        typingMessage.setChatRoomId(chatRoomId);
//        typingMessage.setCreatedAt(LocalDateTime.now());
//
//        sendChatMessage(typingMessage);
//    }
}