package com.finconsgroup.itserr.marketplace.usercommunication.dm.repository;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationMessageSummaryDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ChatMessage;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Repository;

import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

@Repository
public interface ChatMessageRepository extends JpaRepository<ChatMessage, UUID> {

    /**
     * Find a message by its unique message ID
     */
    @NonNull
    Optional<ChatMessage> findById(@NonNull UUID messageId);

    /**
     * Find the messages by conversation id and its unique message IDs
     */
    @NonNull
    List<ChatMessage> findByIdInAndConversationId(@NonNull Collection<UUID> ids, @NonNull UUID conversationId);

    /**
     * Find all messages for a specific conversation, ordered by timestamp
     */
    List<ChatMessage> findByConversationIdOrderByCreatedAtAsc(UUID conversationId);

    /**
     * Find the count of messages by conversation id for the provided ids.
     *
     * @param conversationIds the conversation ids
     * @return the list of {@link OutputConversationMessageSummaryDto} containing the counts
     */
    @Query("SELECT new com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationMessageSummaryDto(cm.conversationId, count(cm)) " +
            "FROM ChatMessage cm " +
            "WHERE cm.conversationId in (:conversationIds) " +
            "GROUP BY cm.conversationId")
    List<OutputConversationMessageSummaryDto> countByConversationId(Set<UUID> conversationIds);

    /**
     * Find the latest messages for a specific conversation, ordered by creation timestamp descending order
     */
    @Query(value = "WITH messages AS (SELECT cm.*, " +
            "rank() OVER (PARTITION BY cm.conversation_id ORDER by cm.created_at DESC) message_rank " +
            "FROM chat_messages cm " +
            "WHERE cm.conversation_id IN (:conversationIds))" +
            "SELECT messages.* FROM messages WHERE message_rank = 1",
            nativeQuery = true)
    List<ChatMessage> findLatestByConversationIdIn(Set<UUID> conversationIds);

    /**
     * Count the unread messages for a provided conversations
     *
     * @param conversationIds the conversation ids
     */
    @Query(value = """
            SELECT new com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationMessageSummaryDto(
            cm.conversationId, COUNT(cm)) FROM ChatMessage cm
            WHERE cm.conversationId IN (:conversationIds)
            AND cm.senderId <> :userId
            AND cm.readAt IS NULL
            GROUP BY cm.conversationId
            """)
    List<OutputConversationMessageSummaryDto> countUnreadByConversationIdIn(@Param("userId") UUID userId,
                                                                            @Param("conversationIds") Set<UUID> conversationIds);

    /**
     * Find the earliest unread message for a provided conversations, ordered by creation timestamp
     *
     * @param conversationIds  the conversation ids
     */
    @Query(value = """
                WITH messages AS (SELECT cm.*,
                rank() OVER (PARTITION BY cm.conversation_id ORDER by cm.created_at) message_rank
                FROM chat_messages cm
                WHERE cm.conversation_id IN (:conversationIds)
                AND cm.sender_id <> :userId
                AND cm.read_at IS NULL)
                SELECT messages.* FROM messages WHERE message_rank = 1
            """,
            nativeQuery = true)
    List<ChatMessage> findEarliestUnreadByConversationIdIn(@Param("userId") UUID userId,
                                                           @Param("conversationIds") Set<UUID> conversationIds);

    /**
     * Find messages by sender ID
     */
    List<ChatMessage> findBySenderIdOrderByCreatedAtDesc(UUID senderId);

    /**
     * Find page of messages in a conversation with offset and limit for lazy loading
     */
    Page<ChatMessage> findMessagesByConversationId(UUID conversationId, Pageable pageable);

    /**
     * Find page of messages in a conversation for the provided ids with offset and limit for lazy loading
     */
    List<ChatMessage> findMessagesByConversationIdAndIdIn(UUID conversationId, List<UUID> ids);

    /**
     * Search chat messages by search term in the content
     */
    @Query("""
            SELECT cm FROM ChatMessage cm
            JOIN Conversation c on c.id = cm.conversationId
            WHERE c.active = true
            AND LOWER(cm.content) LIKE LOWER(CONCAT('%', :term, '%'))
            AND EXISTS (
                   SELECT 1 FROM ConversationParticipant cp
                   WHERE cp.conversationId = c.id
                   AND cp.userId = :userId
                   AND cp.active = true
            )""")
    Page<ChatMessage> findByContentContainingIgnoreCase(@Param("userId") UUID userId, @Param("term") String term, Pageable pageable);

    /**
     * Mark a specific message as read by receiver
     */
    @Modifying(flushAutomatically = true)
    @Query("UPDATE ChatMessage cm SET cm.readByReceiver = true, cm.readAt = :readAt " +
            "WHERE cm.id = :messageId AND cm.receiverId = :receiverId")
    int markMessageAsReadByReceiver(@Param("messageId") String messageId, @Param("receiverId") UUID receiverId, @Param("readAt") Instant readAt);

    /**
     * Get all unread messages in a conversation that will be marked as read by a specific user
     */
    @Query("""
            SELECT cm.id FROM ChatMessage cm
            WHERE cm.conversationId = :conversationId AND cm.receiverId = :receiverId
            AND cm.readByReceiver = false
            """)
    List<UUID> getConversationMessagesToMarkAsRead(@Param("conversationId") UUID conversationId,
                                                   @Param("receiverId") UUID receiverId,
                                                   @Param("readAt") Instant readAt);

    /**
     * Mark all unread messages in a conversation as read by a specific user
     */
    @Modifying(flushAutomatically = true)
    @Query("""
            UPDATE ChatMessage cm SET cm.readByReceiver = true, cm.readAt = :readAt
            WHERE cm.conversationId = :conversationId AND cm.receiverId = :receiverId
            AND cm.readByReceiver = false
            """)
    int markConversationMessagesAsRead(@Param("conversationId") UUID conversationId,
                                       @Param("receiverId") UUID receiverId,
                                       @Param("readAt") Instant readAt);


    /**
     * Get all unread messages in a conversation that will be marked as read by a specific user upto the created at timestamp.
     */
    @Query("""
            SELECT cm.id FROM ChatMessage cm
            WHERE cm.conversationId = :conversationId AND cm.receiverId = :receiverId
            AND cm.readByReceiver = false AND cm.createdAt <= :uptoCreatedAt
            """)
    List<UUID> getConversationMessagesUptoCreatedAtToMarkAsRead(@Param("conversationId") UUID conversationId,
                                                                @Param("uptoCreatedAt") Instant uptoCreatedAt,
                                                                @Param("receiverId") UUID receiverId,
                                                                @Param("readAt") Instant readAt);

    /**
     * Mark all unread messages in a conversation as read by a specific user upto the created at timestamp.
     */
    @Modifying(flushAutomatically = true)
    @Query("""
            UPDATE ChatMessage cm SET cm.readByReceiver = true, cm.readAt = :readAt
            WHERE cm.conversationId = :conversationId AND cm.receiverId = :receiverId
            AND cm.readByReceiver = false AND cm.createdAt <= :uptoCreatedAt
            """)
    int markConversationMessagesUptoCreatedAtAsRead(@Param("conversationId") UUID conversationId,
                                                    @Param("uptoCreatedAt") Instant uptoCreatedAt,
                                                    @Param("receiverId") UUID receiverId,
                                                    @Param("readAt") Instant readAt);

    /**
     * Get all unread messages in a conversation that will be marked as read by a specific user for the passed ids.
     */
    @Query("""
            SELECT cm.id FROM ChatMessage cm
            WHERE cm.conversationId = :conversationId AND cm.receiverId = :receiverId
            AND cm.readByReceiver = false AND cm.id IN (:ids)
            """)
    List<UUID> getConversationMessagesByIdInToMarkAsRead(@Param("conversationId") UUID conversationId,
                                                         @Param("ids") Set<UUID> ids,
                                                         @Param("receiverId") UUID receiverId,
                                                         @Param("readAt") Instant readAt);

    /**
     * Mark all unread messages in a conversation as read by a specific user for the passed ids.
     */
    @Modifying
    @Query("""
            UPDATE ChatMessage cm SET cm.readByReceiver = true, cm.readAt = :readAt
            WHERE cm.conversationId = :conversationId AND cm.receiverId = :receiverId
            AND cm.readByReceiver = false AND cm.id IN (:ids)
            """)
    int markConversationMessagesByIdInAsRead(@Param("conversationId") UUID conversationId,
                                             @Param("ids") Set<UUID> ids,
                                             @Param("receiverId") UUID receiverId,
                                             @Param("readAt") Instant readAt);

    /**
     * Sets the message index for all the messages that do not have the message index set
     * and the newly calculated index for the conversation
     */
    @Modifying
    @Query(value = """
              with max_index as (
              select
                conversation_id,
                coalesce(max(message_index), 0) as max_index
              from
                chat_messages
              where
                conversation_id = :conversationId
                and message_index is not null
              group by
                conversation_id
              ),
              messages_with_index as (
              select
                cm.id,
                cm.conversation_id,
                row_number() over(partition by cm.conversation_id order by cm.created_at, cm.id) as message_index,
                coalesce(max_index.max_index, 0) as max_index
              from
                chat_messages cm
              left join max_index on
                max_index.conversation_id = cm.conversation_id
              where
                cm.conversation_id = :conversationId
                and message_index is null
              )
              update
                chat_messages
              set
                message_index = messages_with_index.max_index + messages_with_index.message_index
              from
                messages_with_index
              where
                chat_messages.conversation_id = :conversationId
                and chat_messages.id = messages_with_index.id
            """, nativeQuery = true)
    int updateMessageIndex(@Param("conversationId") UUID conversationId);
}