package com.finconsgroup.itserr.marketplace.usercommunication.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.ConversationType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.EnumType;
import jakarta.persistence.Enumerated;
import jakarta.persistence.Table;
import jakarta.validation.constraints.NotNull;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.util.UUID;

/**
 * Entity representing a conversation that can be:
 * - DIRECT: 1-on-1 chat between two users (LinkedIn-style)
 * - GROUP: Private group chat with invited members
 */
@Entity
@Table(name = "conversations")
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class Conversation extends AbstractUUIDEntity {

    @Enumerated(EnumType.STRING)
    @Column(name = "conversation_type", nullable = false)
    @NotNull(message = "Conversation type cannot be null")
    private ConversationType conversationType;

    @Column(name = "name")
    private String name; // Optional for DIRECT, required for GROUP

    @Column(name = "description", columnDefinition = "TEXT")
    private String description;

    @Column(name = "created_by", nullable = false)
    @NotNull(message = "Created by cannot be blank")
    private UUID createdBy;

    @Column(name = "created_at", nullable = false)
    private Instant createdAt;

    @Column(name = "last_activity", nullable = false)
    private Instant lastActivity;

    @Column(name = "active", nullable = false)
    private boolean active;

    @Column(name = "is_private", nullable = false)
    private boolean privateFlag; // true for DIRECT and GROUP

    @Column(name = "max_participants")
    private Integer maxParticipants; // null for DIRECT (always 2), configurable for others

    // For DIRECT conversations, store the two participant IDs for quick lookup
    @Column(name = "participant_one_id")
    private UUID participantOneId;

    @Column(name = "participant_two_id")
    private UUID participantTwoId;

    public Conversation(UUID conversationId, ConversationType conversationType, UUID createdBy) {
        this();
        this.setId(conversationId);
        this.conversationType = conversationType;
        this.createdBy = createdBy;
        
        // Set defaults based on conversation type
        switch (conversationType) {
            case DIRECT:
                this.privateFlag = true;
                this.maxParticipants = 2;
                break;
            case GROUP:
                this.privateFlag = true;
                this.maxParticipants = 50; // Default for groups
                break;
        }
    }

    @Override
    public void prePersist() {
        super.prePersist();
        this.createdAt = Instant.now();
        this.lastActivity = Instant.now();
        this.active = true;
    }

    // Static factory methods for different conversation types
    public static Conversation createDirectConversation(UUID conversationId, UUID user1Id, UUID user2Id) {
        Conversation conversation = new Conversation(conversationId, ConversationType.DIRECT, user1Id);
        conversation.setParticipantOneId(user1Id);
        conversation.setParticipantTwoId(user2Id);
        conversation.setName(null); // Direct conversations don't need names
        return conversation;
    }

    public static Conversation createGroupConversation(UUID conversationId, String name, UUID createdBy) {
        Conversation conversation = new Conversation(conversationId, ConversationType.GROUP, createdBy);
        conversation.setName(name);
        return conversation;
    }


    public boolean isDirect() {
        return conversationType == ConversationType.DIRECT;
    }

    public boolean isGroup() {
        return conversationType == ConversationType.GROUP;
    }


    /**
     * For direct conversations, check if a user is one of the two participants
     */
    public boolean isDirectParticipant(UUID userId) {
        return isDirect() && (userId.equals(participantOneId) || userId.equals(participantTwoId));
    }

    /**
     * For direct conversations, get the other participant's ID
     */
    public UUID getOtherParticipantId(UUID userId) {
        if (!isDirect()) {
            return null;
        }
        if (userId.equals(participantOneId)) {
            return participantTwoId;
        } else if (userId.equals(participantTwoId)) {
            return participantOneId;
        }
        return null;
    }
}