package com.finconsgroup.itserr.marketplace.usercommunication.dm.component;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.bean.ChatMessageApplicationEvent;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ChatMessage;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.ChatMessageService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.ApplicationListener;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import static com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessagingType.SPRING_EVENTS;

/**
 * Application Event listener for {@link ChatMessageApplicationEvent}s to handle the received chat messages
 * and user notifications.
 */
@ConditionalOnProperty(value = "type", prefix = "user-communication.dm.messaging", havingValue = SPRING_EVENTS)
@Component
@RequiredArgsConstructor
@Slf4j
public class ChatApplicationEventListener implements ApplicationListener<ChatMessageApplicationEvent> {

    private final ChatMessageService chatMessageService;

    @Override
    public void onApplicationEvent(@NonNull ChatMessageApplicationEvent event) {
        ChatMessage message = event.getChatMessage();
        if (event.isUserNotification()) {
            log.info("Received user notification: {} for user: {}",
                    message.getId(), message.getReceiverId());

            // Store notification in the database
            chatMessageService.handleUserNotification(message);
        } else {
            log.info("Received chat message: {} from user: {} in conversation: {}",
                    message.getId(), message.getSenderId(), message.getConversationId());
            chatMessageService.handleChatMessage(message);
        }
    }

}
