package com.finconsgroup.itserr.marketplace.usercommunication.dm.api;

import com.finconsgroup.itserr.marketplace.core.web.utils.OpenApiUtils;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputCreateGroupConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputInviteUserDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputPromoteUserDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputRemoveUserDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.Conversation;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;

/**
 * API contract for group conversation related endpoints.
 * Contains endpoint mappings and parameter annotations.
 */
@Tag(name = "Group Conversation", description = "Operations related to group conversations")
@SecurityRequirement(name = "BearerAuth")
@ApiResponses(value = {
        @ApiResponse(responseCode = "500", ref = OpenApiUtils.RESPONSE_INTERNAL_SERVER_ERROR)
})
public interface GroupConversationApi {

    @Operation(summary = "Create group conversation", description = "Create a new group conversation.",
            hidden = true)
    @ApiResponses(value = {
            @ApiResponse(responseCode = "201", description = "Group conversation created"),
            @ApiResponse(responseCode = "400", description = "Invalid request")
    })
    @PostMapping(value = "/user-communication/conversations/groups", consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE)
    ResponseEntity<Conversation> createGroupConversation(@RequestBody InputCreateGroupConversationDto request);

    @Operation(summary = "Invite user to group", description = "Invite a user to join a group conversation.",
            hidden = true)
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "User invited"),
            @ApiResponse(responseCode = "404", description = "Conversation or user not found"),
            @ApiResponse(responseCode = "400", description = "Invalid request")
    })
    @PostMapping(value = "/user-communication/conversations/groups/{conversationId}/invite", consumes = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void inviteUserToGroup(@PathVariable UUID conversationId,
                           @RequestBody InputInviteUserDto request);

    @Operation(summary = "Remove user from group", description = "Remove a user from a group conversation.",
            hidden = true)
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "User removed"),
            @ApiResponse(responseCode = "404", description = "Conversation or user not found"),
            @ApiResponse(responseCode = "400", description = "Invalid request")
    })
    @PostMapping(value = "/user-communication/conversations/groups/{conversationId}/remove",
            consumes = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void removeUserFromGroup(@PathVariable UUID conversationId,
                             @RequestBody InputRemoveUserDto request);

    @Operation(summary = "Promote user to admin", description = "Promote a participant to admin in a group conversation.",
            hidden = true)
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "User promoted"),
            @ApiResponse(responseCode = "404", description = "Conversation or user not found"),
            @ApiResponse(responseCode = "400", description = "Invalid request")
    })
    @PostMapping(value = "/user-communication/conversations/groups/{conversationId}/promote",
            consumes = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void promoteUserToAdmin(@PathVariable UUID conversationId,
                            @RequestBody InputPromoteUserDto request);

}
