package com.finconsgroup.itserr.marketplace.usercommunication.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2AuthorizationException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ValidationException;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.Conversation;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ConversationParticipant;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.ParticipantRole;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.ConversationParticipantRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.ConversationRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.GroupConversationService;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.PreConditionService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.Optional;
import java.util.UUID;

/**
 * Service implementation for {@link GroupConversationService}.
 * Contains business rules for participants, permissions, read receipts, and stats.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class GroupConversationServiceImpl implements GroupConversationService {

    private final PreConditionService preConditionService;
    private final ConversationRepository conversationRepository;
    private final ConversationParticipantRepository participantRepository;

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public Conversation createGroupConversation(String name, String description, UUID createdBy) {
        UUID conversationId = UUID.randomUUID();
        Conversation conversation = Conversation.createGroupConversation(conversationId, name, createdBy);
        conversation.setDescription(description);
        conversation = conversationRepository.save(conversation);
        ConversationParticipant owner = ConversationParticipant.createOwner(conversationId, createdBy);
        participantRepository.save(owner);
        log.info("Created new group conversation '{}' by {}: {}", name, createdBy, conversationId);
        return conversation;
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public void inviteUserToGroup(UUID conversationId, UUID userId, UUID invitedBy) {
        Conversation conversation = preConditionService.checkUserAndConversation(invitedBy,
                conversationId);
        if (!participantRepository.hasAdminPrivileges(conversationId, invitedBy)) {
            log.warn("User {} does not have admin privileges to invite users to conversation {}",
                    invitedBy, conversationId);
            throw new WP2AuthorizationException("Unauthorized");
        }
        long currentParticipants = participantRepository.countByConversationIdAndActiveIsTrue(conversationId);
        if (currentParticipants >= conversation.getMaxParticipants()) {
            log.warn("Conversation {} has reached maximum participants limit", conversationId);
            throw new WP2ValidationException("Conversation has reached maximum participants limit");
        }
        ConversationParticipant participant = ConversationParticipant.createInvitedMember(conversationId, userId, invitedBy);
        participantRepository.save(participant);
        log.info("User {} invited to group conversation {} by {}", userId, conversationId, invitedBy);
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public boolean removeUserFromGroup(UUID conversationId, UUID userId, UUID removedBy) {
        Optional<Conversation> conversationOpt = conversationRepository.findById(conversationId);
        if (conversationOpt.isEmpty() || !conversationOpt.get().isGroup()) {
            return false;
        }
        if (!removedBy.equals(userId) && !participantRepository.hasAdminPrivileges(conversationId, removedBy)) {
            log.warn("User {} does not have permission to remove user {} from conversation {}",
                    removedBy, userId, conversationId);
            return false;
        }
        Optional<ConversationParticipant> participantOpt = participantRepository.findByConversationIdAndUserId(conversationId, userId);
        if (participantOpt.isPresent() && participantOpt.get().isOwner() && !removedBy.equals(userId)) {
            log.warn("Cannot remove owner {} from conversation {} unless they remove themselves",
                    userId, conversationId);
            return false;
        }
        participantRepository.removeParticipant(conversationId, userId);
        log.info("User {} removed from group conversation {} by {}", userId, conversationId, removedBy);
        return true;
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public void promoteUserToAdmin(UUID conversationId, UUID userId, UUID promotedBy) {
        if (!participantRepository.hasAdminPrivileges(conversationId, promotedBy)) {
            log.warn("User {} does not have admin privileges to promote users in conversation {}",
                    promotedBy, conversationId);
            throw new WP2AuthorizationException("Unauthorized");
        }
        Optional<ConversationParticipant> participantOpt = participantRepository.findByConversationIdAndUserId(conversationId, userId);
        if (participantOpt.isEmpty() || !participantOpt.get().isActive()) {
            log.warn("User {} is not an active participant in conversation {}", userId, conversationId);
            throw new WP2ValidationException("User is not an active participant");
        }
        ConversationParticipant participant = participantOpt.get();
        if (participant.isAdmin() || participant.isOwner()) {
            log.warn("User {} is already an admin or owner in conversation {}", userId, conversationId);
            return;
        }
        participantRepository.updateParticipantRole(conversationId, userId, ParticipantRole.ADMIN);
        log.info("User {} promoted to admin in conversation {} by {}", userId, conversationId, promotedBy);
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public boolean isUserParticipant(UUID conversationId, UUID userId) {
        return participantRepository.existsByConversationIdAndUserIdAndActiveIsTrue(conversationId, userId);
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public boolean hasAdminPrivileges(UUID conversationId, UUID userId) {
        return participantRepository.hasAdminPrivileges(conversationId, userId);
    }
}
