package com.finconsgroup.itserr.marketplace.usercommunication.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.ParticipantRole;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.EnumType;
import jakarta.persistence.Enumerated;
import jakarta.persistence.Table;
import jakarta.persistence.UniqueConstraint;
import jakarta.validation.constraints.NotNull;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.util.UUID;

/**
 * Entity representing a participant in a conversation with their role and status
 */
@Entity
@Table(name = "conversation_participants",
        uniqueConstraints = @UniqueConstraint(columnNames = {"conversation_id", "user_id"}))
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(of = {"conversationId", "userId"}, callSuper = false)
@NoArgsConstructor
public class ConversationParticipant extends AbstractUUIDEntity {

    @Column(name = "conversation_id", nullable = false)
    @NotNull(message = "Conversation ID cannot be blank")
    private UUID conversationId;

    @Column(name = "user_id", nullable = false)
    @NotNull(message = "User ID cannot be blank")
    private UUID userId;

    @Enumerated(EnumType.STRING)
    @Column(name = "role", nullable = false)
    @NotNull(message = "Participant role cannot be null")
    private ParticipantRole role;

    @Column(name = "joined_at", nullable = false)
    private Instant joinedAt;

    @Column(name = "invited_at")
    private Instant invitedAt;

    @Column(name = "invited_by")
    private UUID invitedBy;

    @Column(name = "is_active", nullable = false)
    private boolean active; // false if user left the conversation

    @Column(name = "is_muted", nullable = false)
    private boolean muted; // user muted notifications for this conversation

    @Column(name = "last_read_at")
    private Instant lastReadAt; // for unread message tracking

    @Column(name = "notification_enabled", nullable = false)
    private boolean notificationEnabled;

    public ConversationParticipant(UUID conversationId, UUID userId, ParticipantRole role) {
        this();
        this.conversationId = conversationId;
        this.userId = userId;
        this.role = role;
    }

    public ConversationParticipant(UUID conversationId, UUID userId, ParticipantRole role, UUID invitedBy) {
        this(conversationId, userId, role);
        this.invitedBy = invitedBy;
        this.invitedAt = Instant.now();
    }

    @Override
    public void prePersist() {
        super.prePersist();
        this.joinedAt = Instant.now();
        this.active = true;
        this.muted = false;
        this.notificationEnabled = true;
    }

    // Static factory methods
    public static ConversationParticipant createOwner(UUID conversationId, UUID userId) {
        return new ConversationParticipant(conversationId, userId, ParticipantRole.OWNER);
    }

    public static ConversationParticipant createMember(UUID conversationId, UUID userId) {
        return new ConversationParticipant(conversationId, userId, ParticipantRole.MEMBER);
    }

    public static ConversationParticipant createInvitedMember(UUID conversationId, UUID userId, UUID invitedBy) {
        return new ConversationParticipant(conversationId, userId, ParticipantRole.MEMBER, invitedBy);
    }

    public static ConversationParticipant createAdmin(UUID conversationId, UUID userId, UUID promotedBy) {
        ConversationParticipant participant = new ConversationParticipant(conversationId, userId, ParticipantRole.ADMIN);
        participant.setInvitedBy(promotedBy);
        return participant;
    }

    // Utility methods
    public boolean isOwner() {
        return role == ParticipantRole.OWNER;
    }

    public boolean isAdmin() {
        return role == ParticipantRole.ADMIN;
    }

    public boolean isMember() {
        return role == ParticipantRole.MEMBER;
    }

    public boolean hasAdminPrivileges() {
        return isOwner() || isAdmin();
    }

    public void promoteToAdmin() {
        if (isMember()) {
            this.role = ParticipantRole.ADMIN;
        }
    }

    public void demoteToMember() {
        if (isAdmin()) {
            this.role = ParticipantRole.MEMBER;
        }
    }

    public void leave() {
        this.active = false;
    }

    public void rejoin() {
        this.active = true;
        this.joinedAt = Instant.now();
    }

    public void markAsRead() {
        this.lastReadAt = Instant.now();
    }
}