package com.finconsgroup.itserr.marketplace.usercommunication.dm.api;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputChatMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputTypingMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputChatMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.MessageType;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.security.WebSocketAuthentication;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.messaging.handler.annotation.DestinationVariable;
import org.springframework.messaging.handler.annotation.Payload;
import org.springframework.messaging.simp.SimpMessageHeaderAccessor;
import org.springframework.security.core.annotation.AuthenticationPrincipal;

import java.util.UUID;

/**
 * API contract for WebSocketChatController. Annotated for documentation purposes
 * to describe STOMP destinations and behaviors even though these are not REST endpoints.
 */
public interface WebSocketChatApi {

    @Operation(summary = "Send message to conversation (WebSocket)",
            description = "STOMP send to /app/conversation/{conversationId}. The message is then broadcast to /topic/conversation/{conversationId}.")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Message processed and broadcasted"),
            @ApiResponse(responseCode = "403", description = "User not a participant in the conversation"),
            @ApiResponse(responseCode = "400", description = "Invalid payload")
    })
    void sendMessage(@DestinationVariable UUID conversationId,
                     @Payload InputChatMessageDto chatMessage,
                     @AuthenticationPrincipal WebSocketAuthentication principal,
                     SimpMessageHeaderAccessor headerAccessor);

    @Operation(summary = "Join conversation (WebSocket)",
            description = "STOMP send to /app/conversation/{conversationId}/join. Announces a JOIN event to /topic/conversation/{conversationId}.")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Join processed and announced"),
            @ApiResponse(responseCode = "404", description = "Conversation not found"),
            @ApiResponse(responseCode = "403", description = "User not allowed to join (e.g., private group without invitation)")
    })
    void addUser(@DestinationVariable UUID conversationId,
                 @Payload InputChatMessageDto chatMessage,
                 @AuthenticationPrincipal WebSocketAuthentication principal,
                 SimpMessageHeaderAccessor headerAccessor);

    @Operation(summary = "Leave conversation (WebSocket)",
            description = "STOMP send to /app/conversation/{conversationId}/leave. Announces a LEAVE event to /topic/conversation/{conversationId}.")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Leave processed and announced"),
            @ApiResponse(responseCode = "403", description = "User not a participant"),
            @ApiResponse(responseCode = "400", description = "Invalid payload")
    })
    void removeUser(@DestinationVariable UUID conversationId,
                    @Payload InputChatMessageDto chatMessage,
                    @AuthenticationPrincipal WebSocketAuthentication principal,
                    SimpMessageHeaderAccessor headerAccessor);

    @Operation(summary = "Typing indicator (WebSocket)",
            description = "STOMP send to /app/conversation/{conversationId}/typing. Broadcasts typing status to /topic/conversation/{conversationId}/typing.")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Typing indicator broadcasted"),
            @ApiResponse(responseCode = "403", description = "User not a participant"),
            @ApiResponse(responseCode = "400", description = "Invalid payload")
    })
    void sendTypingIndicator(@DestinationVariable UUID conversationId,
                             @Payload InputTypingMessageDto inputTypingMessageDto,
                             @AuthenticationPrincipal WebSocketAuthentication principal);

    @Operation(summary = "Send direct message (WebSocket)",
            description = "STOMP send to /app/conversation/direct/{receiverId}. Creates/uses a direct conversation and broadcasts to corresponding /topic and user queue.")
    @ApiResponses(value = {
        @ApiResponse(responseCode = "200", description = "Direct message processed"),
        @ApiResponse(responseCode = "400", description = "Invalid payload")
    })
    void sendDirectMessage(@DestinationVariable UUID conversationId,
                           @Payload InputChatMessageDto chatMessage,
                           @AuthenticationPrincipal WebSocketAuthentication principal);

    @Operation(summary = "Broadcast system message (WebSocket)",
            description = "Programmatic utility to broadcast a system message to all users in a conversation.")
    @ApiResponses(value = {
        @ApiResponse(responseCode = "200", description = "System message broadcasted")
    })
    void broadcastSystemMessage(UUID conversationId, String content);

    @Operation(summary = "Send user notification (WebSocket)",
            description = "Programmatic utility to send a notification to a specific user queue.")
    @ApiResponses(value = {
        @ApiResponse(responseCode = "200", description = "Notification sent")
    })
    void sendUserNotification(UUID userId, String content, MessageType messageType);

    @Operation(summary = "Send invitation notification (WebSocket)",
            description = "Programmatic utility to send an invitation notification to a user for a group conversation.")
    @ApiResponses(value = {
        @ApiResponse(responseCode = "200", description = "Invitation sent")
    })
    void sendInvitationNotification(UUID conversationId,
                                    UUID invitedUserId,
                                    @AuthenticationPrincipal WebSocketAuthentication principal);
}
