package com.finconsgroup.itserr.marketplace.usercommunication.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2AuthorizationException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.SecurityConstants;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.Conversation;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.MessageType;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.ConversationParticipantRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.ConversationRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.PreConditionService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.UUID;

/**
 * Default implementation for {@link PreConditionService}
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultPreConditionService implements PreConditionService {

    private final ConversationRepository conversationRepository;
    private final ConversationParticipantRepository conversationParticipantRepository;

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public Conversation checkConversation(UUID conversationId) {
        return conversationRepository.findById(conversationId)
                .filter(Conversation::isActive)
                .orElseThrow(() -> new WP2ResourceNotFoundException(conversationId));
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public Conversation checkUserAndConversation(UUID userId, UUID conversationId) {
        @SuppressWarnings("squid:S6809")
        Conversation conversation = checkConversation(conversationId);
        checkUserAndConversation(userId, conversation);
        return conversation;
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public Conversation checkUserAndConversationAndMessage(UUID userId, UUID conversationId, MessageType messageType) {
        @SuppressWarnings("squid:S6809")
        Conversation conversation = checkConversation(conversationId);
        switch (messageType) {
            case NOTIFICATION -> {
                if (!SecurityConstants.SYSTEM_USER_ID.equals(userId)) {
                    log.warn("User {} attempted to send message with type {} to conversation {} but is not allowed",
                            userId, messageType, conversationId);
                    throw new WP2AuthorizationException("Not Authorized");
                }
            }
            default -> checkUserAndConversation(userId, conversation);
        }
        return conversation;
    }

    public void checkUserAndConversation(UUID userId, Conversation conversation) {
        UUID conversationId = conversation.getId();
        // a user is allowed to access a conversation if an active participant
        if (!conversationParticipantRepository.existsByConversationIdAndUserIdAndActiveIsTrue(conversationId, userId)) {
            log.warn("User {} attempted to access conversation {} but is not allowed",
                    userId, conversationId);
            throw new WP2AuthorizationException("Not Authorized");
        }
    }

}
