package com.finconsgroup.itserr.marketplace.usercommunication.dm.security.config;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenVerifier;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.configuration.properties.UserCommunicationSecurityProperties;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessageDestinations;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.security.AuthenticationChannelInterceptor;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.security.SubscriptionAuthorizationChannelInterceptor;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.PreConditionService;
import lombok.RequiredArgsConstructor;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.lang.NonNull;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageChannel;
import org.springframework.messaging.handler.invocation.HandlerMethodArgumentResolver;
import org.springframework.messaging.simp.config.ChannelRegistration;
import org.springframework.messaging.support.ChannelInterceptor;
import org.springframework.security.authorization.AuthorizationEventPublisher;
import org.springframework.security.authorization.AuthorizationManager;
import org.springframework.security.authorization.SpringAuthorizationEventPublisher;
import org.springframework.security.messaging.access.intercept.AuthorizationChannelInterceptor;
import org.springframework.security.messaging.access.intercept.MessageMatcherDelegatingAuthorizationManager;
import org.springframework.security.messaging.context.AuthenticationPrincipalArgumentResolver;
import org.springframework.security.messaging.context.SecurityContextChannelInterceptor;
import org.springframework.web.socket.config.annotation.WebSocketMessageBrokerConfigurer;

import java.util.List;
import java.util.Optional;

import static com.finconsgroup.itserr.marketplace.core.web.security.jwt.SecurityRoles.MEMBER_ROLE;
import static org.springframework.messaging.simp.SimpMessageType.DISCONNECT;
import static org.springframework.messaging.simp.SimpMessageType.MESSAGE;
import static org.springframework.messaging.simp.SimpMessageType.SUBSCRIBE;

@Configuration
@RequiredArgsConstructor
public class WebSocketSecurityConfig implements WebSocketMessageBrokerConfigurer {

    @Configuration
    public static class AuthorizationConfiguration {

        @Bean
        @SuppressWarnings("squid:S1452")
        public AuthorizationManager<Message<?>> messageAuthorizationManager(MessageDestinations messageDestinations) {
            return MessageMatcherDelegatingAuthorizationManager.builder()
                    // allow all as connection is automatically closed for unauthenticated users,
                    // which should be allowed
                    .simpTypeMatchers(DISCONNECT).permitAll()
                    .nullDestMatcher().authenticated()
                    // only allow authenticated VRE Member users
                    // need to use hasAuthority as, hasRole adds role prefix i.e. ROLE_
                    .simpSubscribeDestMatchers(messageDestinations.getUserErrors()).hasAuthority(MEMBER_ROLE)
                    // only allow authenticated VRE Member users
                    .simpMessageDestMatchers("/app/**").hasAuthority(MEMBER_ROLE)
                    // only allow authenticated VRE Member users to access the user queues and all topics
                    .simpSubscribeDestMatchers(messageDestinations.getUserDestinationPrefix() + "/**",
                            "/topic/**").hasAuthority(MEMBER_ROLE)
                    .simpTypeMatchers(MESSAGE, SUBSCRIBE).denyAll()
                    .anyMessage().denyAll()
                    .build();
        }

        @Bean
        public AuthorizationChannelInterceptor authorizationChannelInterceptor(ApplicationContext applicationContext,
                                                                               AuthorizationManager<Message<?>> authorizationManager) {
            AuthorizationChannelInterceptor authorizationChannelInterceptor = new AuthorizationChannelInterceptor(authorizationManager);
            AuthorizationEventPublisher publisher = new SpringAuthorizationEventPublisher(applicationContext);
            authorizationChannelInterceptor.setAuthorizationEventPublisher(publisher);
            return authorizationChannelInterceptor;
        }

        @Bean
        public SubscriptionAuthorizationChannelInterceptor subscriptionAuthorizationChannelInterceptor(PreConditionService preConditionService,
                                                                                                       MessageDestinations messageDestinations) {
            return new SubscriptionAuthorizationChannelInterceptor(preConditionService, messageDestinations);
        }
    }

    private final Optional<JwtTokenVerifier> jwtTokenVerifier;
    private final AuthorizationChannelInterceptor authorizationChannelInterceptor;
    private final SubscriptionAuthorizationChannelInterceptor subscriptionAuthorizationChannelInterceptor;
    private final UserCommunicationSecurityProperties securityProperties;

    @Override
    public void addArgumentResolvers(List<HandlerMethodArgumentResolver> argumentResolvers) {
        argumentResolvers.add(new AuthenticationPrincipalArgumentResolver());
    }

    @Override
    public void configureClientInboundChannel(ChannelRegistration registration) {
        AuthenticationChannelInterceptor authenticationChannelInterceptor = new AuthenticationChannelInterceptor(
                securityProperties.getAudience(), jwtTokenVerifier.orElse(null));
        registration.interceptors(authenticationChannelInterceptor,
                new SecurityContextChannelInterceptor(), authorizationChannelInterceptor,
                subscriptionAuthorizationChannelInterceptor);
    }

    @Override
    public void configureClientOutboundChannel(ChannelRegistration registration) {
        registration.interceptors(new ChannelInterceptor() {
            @Override
            public Message<?> preSend(@NonNull Message<?> message, @NonNull MessageChannel channel) {
                return message;
            }
        });
    }
}
