package com.finconsgroup.itserr.marketplace.usercommunication.dm.repository;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationMessageSummaryDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ChatMessage;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.MessageReadReceipt;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

@Repository
public interface MessageReadReceiptRepository extends JpaRepository<MessageReadReceipt, UUID> {

    /**
     * Find all read receipts for provided message ids
     */
    List<MessageReadReceipt> findByMessageIdIn(Set<UUID> messageIds);

    /**
     * Get all messages in a conversation to mark as read by a user (bulk operation)
     */
    @Modifying
    @Query(value = """
            SELECT cm.id
            FROM chat_messages cm
            WHERE cm.conversation_id = :conversationId
            AND cm.sender_id != :userId
            AND NOT EXISTS (SELECT 1 FROM message_read_receipts mrr WHERE mrr.message_id = cm.id AND mrr.user_id = :userId)
            """, nativeQuery = true)
    List<UUID> getAllMessagesToMarkAsReadByUser(@Param("userId") UUID userId,
                                                @Param("conversationId") UUID conversationId);

    /**
     * Mark all messages in a conversation as read by a user (bulk operation)
     * This creates read receipts for all unread messages
     */
    @Modifying
    @Query(value = """
            INSERT INTO message_read_receipts (message_id, user_id, conversation_id, read_at, created_at)
            SELECT cm.id, :userId, :conversationId, :readAt, :readAt
            FROM chat_messages cm
            WHERE cm.conversation_id = :conversationId
            AND cm.sender_id != :userId
            AND NOT EXISTS (SELECT 1 FROM message_read_receipts mrr WHERE mrr.message_id = cm.id AND mrr.user_id = :userId)
            """, nativeQuery = true)
    int markAllMessagesAsReadByUser(@Param("userId") UUID userId, 
                                   @Param("conversationId") UUID conversationId, 
                                   @Param("readAt") Instant readAt);

    /**
     * Get all messages in a conversation to mark as read by a user (bulk operation) upto the created at value provided.
     */
    @Modifying
    @Query(value = """
            SELECT cm.id
            FROM chat_messages cm
            WHERE cm.conversation_id = :conversationId
            AND cm.sender_id != :userId
            AND cm.created_at <= :uptoCreatedAt
            AND NOT EXISTS (SELECT 1 FROM message_read_receipts mrr WHERE mrr.message_id = cm.id AND mrr.user_id = :userId)
            """, nativeQuery = true)
    List<UUID> getMessagesUptoCreatedAtToMarkAsReadByUser(@Param("userId") UUID userId,
                                                          @Param("conversationId") UUID conversationId,
                                                          @Param("uptoCreatedAt") Instant uptoCreatedAt);

    /**
     * Mark all messages in a conversation as read by a user (bulk operation) upto the created at value provided.
     * This creates read receipts for all relevant unread messages
     */
    @Modifying
    @Query(value = """
            INSERT INTO message_read_receipts (message_id, user_id, conversation_id, read_at, created_at)
            SELECT cm.id, :userId, :conversationId, :readAt, :readAt
            FROM chat_messages cm
            WHERE cm.conversation_id = :conversationId
            AND cm.sender_id != :userId
            AND cm.created_at <= :uptoCreatedAt
            AND NOT EXISTS (SELECT 1 FROM message_read_receipts mrr WHERE mrr.message_id = cm.id AND mrr.user_id = :userId)
            """, nativeQuery = true)
    int markMessagesUptoCreatedAtAsReadByUser(@Param("userId") UUID userId,
                                              @Param("conversationId") UUID conversationId,
                                              @Param("uptoCreatedAt") Instant uptoCreatedAt,
                                              @Param("readAt") Instant readAt);

    /**
     * Gets all messages in a conversation to mark as read by a user (bulk operation) for the provided ids.
     */
    @Query(value = """
            SELECT cm.id
            FROM chat_messages cm
            WHERE cm.conversation_id = :conversationId
            AND cm.sender_id != :userId
            AND cm.id IN (:ids)
            AND NOT EXISTS (SELECT 1 FROM message_read_receipts mrr WHERE mrr.message_id = cm.id AND mrr.user_id = :userId)
            """, nativeQuery = true)
    List<UUID> getMessagesByIdInToMarkAsReadByUser(@Param("userId") UUID userId,
                                                   @Param("conversationId") UUID conversationId,
                                                   @Param("ids") Set<UUID> ids);

    /**
     * Mark all messages in a conversation as read by a user (bulk operation) for the provided ids.
     * This creates read receipts for all relevant unread messages
     */
    @Modifying
    @Query(value = """
            INSERT INTO message_read_receipts (message_id, user_id, conversation_id, read_at, created_at)
            SELECT cm.id, :userId, :conversationId, :readAt, :readAt
            FROM chat_messages cm
            WHERE cm.conversation_id = :conversationId
            AND cm.sender_id != :userId
            AND cm.id IN (:ids)
            AND NOT EXISTS (SELECT 1 FROM message_read_receipts mrr WHERE mrr.message_id = cm.id AND mrr.user_id = :userId)
            """, nativeQuery = true)
    int markMessagesByIdInAsReadByUser(@Param("userId") UUID userId,
                                       @Param("conversationId") UUID conversationId,
                                       @Param("ids") Set<UUID> ids,
                                       @Param("readAt") Instant readAt);


    /**
     * Count the unread messages for a specific conversation, ordered by creation timestamp
     */
    @Query(value = """
                    SELECT new com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationMessageSummaryDto(cm.conversationId, count(cm))
                    FROM ChatMessage cm
                    LEFT JOIN MessageReadReceipt mrr ON mrr.messageId = cm.id AND mrr.userId = :userId
                    WHERE cm.conversationId IN (:conversationIds)
                    AND cm.senderId <> :userId
                    AND mrr.id IS NULL
            """)
    List<OutputConversationMessageSummaryDto> countUnreadByConversationIdIn(@Param("userId") UUID userId,
                                                                            @Param("conversationIds") Set<UUID> conversationIds);

    /**
     * Find the earliest unread message for a specific conversation, ordered by creation timestamp
     */
    @Query(value = """
                    WITH messages AS (SELECT cm.*,
                    rank() OVER (PARTITION BY cm.conversation_id ORDER by cm.created_at) message_rank
                    FROM chat_messages cm
                    left join message_read_receipts mrr on mrr.message_id = cm.id and mrr.user_id = :userId
                    WHERE cm.conversation_id IN (:conversationIds)
                    AND cm.sender_id <> :userId
                    AND mrr.id IS NULL)
                    SELECT messages.* FROM messages WHERE message_rank = 1
            """,
            nativeQuery = true)
    List<ChatMessage> findEarliestUnreadByConversationIdIn(@Param("userId") UUID userId,
                                                           @Param("conversationIds") Set<UUID> conversationIds);
}