package com.finconsgroup.itserr.marketplace.usercommunication.dm.controller;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.api.UserApi;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.User;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.UserRepository;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RestController;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

/**
 * REST controller implementing UserApi for user listing, search and maintenance endpoints.
 * Delegates persistence to UserRepository and logs operations for observability.
 */
@RestController
@RequiredArgsConstructor
@Slf4j
public class UserController implements UserApi {

    private final UserRepository userRepository;
//
//    /**
//     * Get all users ordered by username
//     */
//    @Override
//    public ResponseEntity<List<User>> getAllUsers() {
//        try {
//            List<User> users = userRepository.findAllByOrderByUsernameAsc();
//            log.info("Retrieved {} users", users.size());
//            return ResponseEntity.ok(users);
//        } catch (Exception e) {
//            log.error("Error retrieving users", e);
//            return ResponseEntity.internalServerError().build();
//        }
//    }
//
    /**
     * Get all online users
     */
    @Override
    public ResponseEntity<List<User>> getOnlineUsers() {
        try {
            List<User> onlineUsers = userRepository.findByOnlineTrue();
            log.info("Retrieved {} online users", onlineUsers.size());
            return ResponseEntity.ok(onlineUsers);
        } catch (Exception e) {
            log.error("Error retrieving online users", e);
            return ResponseEntity.internalServerError().build();
        }
    }

    /**
     * Get recently active users (online or active within last hour)
     */
    @Override
    public ResponseEntity<List<User>> getRecentlyActiveUsers() {
        try {
            Instant oneHourAgo = Instant.now().minus(1, ChronoUnit.HOURS);
            List<User> recentUsers = userRepository.findRecentlyActiveUsers(oneHourAgo);
            log.info("Retrieved {} recently active users", recentUsers.size());
            return ResponseEntity.ok(recentUsers);
        } catch (Exception e) {
            log.error("Error retrieving recently active users", e);
            return ResponseEntity.internalServerError().build();
        }
    }

    /**
     * Get user by userId
     */
    @Override
    public ResponseEntity<User> getUserById(UUID userId) {
        Optional<User> user = userRepository.findById(userId);
        if (user.isPresent()) {
            return ResponseEntity.ok(user.get());
        } else {
            log.warn("User not found with userId: {}", userId);
            return ResponseEntity.notFound().build();
        }
    }

//    /**
//     * Search users by username or display name
//     */
//    @Override
//    public ResponseEntity<List<User>> searchUsers(@RequestParam String query) {
//        try {
//            List<User> users = userRepository.findByUsernameContainingIgnoreCase(query);
//            List<User> usersByDisplayName = userRepository.findByDisplayNameContainingIgnoreCase(query);
//
//            // Combine results and remove duplicates
//            users.addAll(usersByDisplayName);
//            users = users.stream().distinct().toList();
//
//            log.info("Found {} users matching query: {}", users.size(), query);
//            return ResponseEntity.ok(users);
//        } catch (Exception e) {
//            log.error("Error searching users with query: {}", query, e);
//            return ResponseEntity.internalServerError().build();
//        }
//    }
//
//    /**
//     * Create or update user
//     */
//    @Override
//    public ResponseEntity<User> createOrUpdateUser(@RequestBody User user) {
//        try {
//            // Check if user already exists
//            Optional<User> existingUser = userRepository.findByUserId(user.getUserId());
//
//            if (existingUser.isPresent()) {
//                // Update existing user
//                User existing = existingUser.get();
//                existing.setUsername(user.getUsername());
//                existing.setEmail(user.getEmail());
//                existing.setDisplayName(user.getDisplayName());
//                existing.setOnline(user.isOnline());
//                User savedUser = userRepository.save(existing);
//                log.info("Updated user: {}", savedUser.getUserId());
//                return ResponseEntity.ok(savedUser);
//            } else {
//                // Create new user
//                User savedUser = userRepository.save(user);
//                log.info("Created new user: {}", savedUser.getUserId());
//                return ResponseEntity.ok(savedUser);
//            }
//        } catch (Exception e) {
//            log.error("Error creating/updating user", e);
//            return ResponseEntity.internalServerError().build();
//        }
//    }

    /**
     * Update user online status
     */
    @Override
    public ResponseEntity<String> updateUserStatus(boolean online) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        userRepository.updateOnlineStatus(userId, online, Instant.now());
        log.info("Updated online status for user {} to {}", userId, online);
        return ResponseEntity.ok("User status updated successfully");
    }
}