package com.finconsgroup.itserr.marketplace.usercommunication.dm.controller;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.api.DirectConversationApi;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputCreateDirectConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.DirectConversationService;
import jakarta.validation.Valid;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.RestController;

import java.util.UUID;

/**
 * REST Controller for managing direct conversations
 * Provides LinkedIn-style direct messaging functionality
 */
@RestController
@RequiredArgsConstructor
@Slf4j
public class DirectConversationController implements DirectConversationApi {

    private final DirectConversationService directConversationService;

    /**
     * Create or get direct conversation between two users (LinkedIn-style)
     */
    @Override
    public ResponseEntity<OutputConversationDto> createDirectConversation(@Valid InputCreateDirectConversationDto request) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        OutputConversationDto conversationDto = directConversationService.createOrGetDirectConversation(
                userId, request.getUser2Id());

        log.info("Direct conversation created/retrieved between {} and {}: {}",
                userId, request.getUser2Id(), conversationDto.getId());

        return ResponseEntity.ok(conversationDto);
    }

    /**
     * Get all direct conversations for a user
     */
    @Override
    public ResponseEntity<OutputConversationDto> getDirectConversationForUserAndReceiver(UUID receiverId) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        OutputConversationDto conversationDto = directConversationService.getDirectConversation(userId, receiverId);
        return ResponseEntity.ok(conversationDto);
    }

}