package com.finconsgroup.itserr.marketplace.usercommunication.dm.constant;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.configuration.properties.WebSocketConfigurationProperties;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.UUID;

/**
 * Utility class for methods related to message destinations.
 */
@Component
@Getter
@Slf4j
public final class MessageDestinations {

    private static final String CONVERSATION_CHILD_SEPARATOR = "--";
    private static final String USER_MESSAGES_SUFFIX = "/queue/messages";
    private static final String USER_NOTIFICATIONS_SUFFIX = "/queue/notifications";
    private static final String USER_INVITATIONS_SUFFIX = "/queue/invitations";

    private final String userDestinationPrefix;
    private final String userErrors;
    private final String topicConversationPrefix;
    private final String topicUserStatus;
    private final String topicUserBroadcast;
    private final String topicUserRegistry;

    public MessageDestinations(WebSocketConfigurationProperties webSocketConfigurationProperties) {
        String destinationPrefix = webSocketConfigurationProperties.isEnableBrokerRelay() ?
                webSocketConfigurationProperties.getRelayDestinationPrefix() : "";
        userDestinationPrefix = "/%suser".formatted(destinationPrefix);
        userErrors = "%s/queue/errors".formatted(userDestinationPrefix);
        topicConversationPrefix = "/topic/%sconversation-".formatted(destinationPrefix);
        topicUserStatus = "/topic/%suser-status".formatted(destinationPrefix);
        topicUserBroadcast = "/topic/%sunresolved-user".formatted(destinationPrefix);
        topicUserRegistry = "/topic/%suser-registry".formatted(destinationPrefix);
    }

    /**
     * @return the user messages suffix
     */
    public String getUserMessagesSuffix() {
        return USER_MESSAGES_SUFFIX;
    }

    /**
     * @return the user notifications suffix
     */
    public String getUserNotificationsSuffix() {
        return USER_NOTIFICATIONS_SUFFIX;
    }

    /**
     * @return the user invitations suffix
     */
    public String getUserInvitationsSuffix() {
        return USER_INVITATIONS_SUFFIX;
    }

    /**
     * Determines if the destination is for user messages.
     *
     * @param destination the destination
     * @return {@code true} if it is, {@code false} otherwise.
     */
    public boolean isUserMessages(String destination) {
        return destination.endsWith(USER_MESSAGES_SUFFIX);
    }

    /**
     * Determines if the destination is for user notifications.
     *
     * @param destination the destination
     * @return {@code true} if it is, {@code false} otherwise.
     */
    public boolean isUserNotifications(String destination) {
        return destination.endsWith(USER_NOTIFICATIONS_SUFFIX);
    }

    /**
     * Determines if the destination is for topic for conversation.
     *
     * @param destination the destination
     * @return {@code true} if it is, {@code false} otherwise.
     */
    public boolean isTopicConversation(String destination) {
        return destination.startsWith(topicConversationPrefix);
    }

    /**
     * Determines if the destination is for user messages or for topic for conversation.
     *
     * @param destination the destination
     * @return {@code true} if it is, {@code false} otherwise.
     */
    public boolean isUserMessagesOrTopicConversation(String destination) {
        return isUserMessages(destination) || isTopicConversation(destination);
    }

    /**
     * Constructs the topic name for the conversation to send the STOMP messages to.
     *
     * @param conversationId the conversation id
     * @return the conversation topic name
     */
    public String getConversationTopic(UUID conversationId) {
        return topicConversationPrefix + conversationId;
    }

    /**
     * Constructs the topic name for the conversation typing indicator to send the STOMP messages to.
     *
     * @param conversationId the conversation id     *
     * @return the conversation typing topic name
     */
    public String getConversationTypingTopic(UUID conversationId) {
        return getConversationChildTopic(conversationId, "typing");
    }

    /**
     * Constructs the topic name for the conversation relate child topic to send the STOMP messages to.
     *
     * @param conversationId the conversation id
     * @param child          the child topic name
     * @return the conversation typing topic name
     */
    public String getConversationChildTopic(UUID conversationId, String child) {
        return topicConversationPrefix + conversationId + CONVERSATION_CHILD_SEPARATOR + child;
    }

    /**
     * Constructs the topic name for the broadcasting messages for users that remain unresolved.
     *
     * @return the user destination broadcast topic name
     */
    public String getUserDestinationBroadcastTopic() {
        return topicUserBroadcast;
    }

    /**
     * Constructs the topic name for the broadcasting user registry messages.
     *
     * @return the user registry broadcast topic name
     */
    public String getUserRegistryBroadcastTopic() {
        return topicUserRegistry;
    }

    /**
     * Determines if the destination is for topic for conversation.
     * If yes, then retrieves and returns the conversation id from the destination name.
     *
     * @param destination the destination
     * @return the conversation id if it is valid.
     * @throws IllegalArgumentException if it does not have a valid conversation id
     */
    @NonNull
    public UUID extractConversationId(@NonNull String destination) {
        if (!isTopicConversation(destination)) {
            throw new IllegalArgumentException("Destination is not a conversation topic");
        }

        String conversationTopicSuffix = destination.replace(topicConversationPrefix, "");
        String[] conversationTopicParts = conversationTopicSuffix.split(CONVERSATION_CHILD_SEPARATOR, 2);
        if (conversationTopicParts.length > 0) {
            try {
                return UUID.fromString(conversationTopicParts[0]);
            } catch (Exception e) {
                throw new IllegalArgumentException("Error converting string " + conversationTopicParts[0] + " to uuid", e);
            }
        } else {
            throw new IllegalArgumentException("Invalid conversationTopicSuffix: " + conversationTopicSuffix);
        }
    }
}
