package com.finconsgroup.itserr.marketplace.usercommunication.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ValidationException;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.component.ConversationHelper;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.Conversation;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ConversationParticipant;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.ConversationParticipantRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.ConversationRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.DirectConversationService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

/**
 * Service implementation for {@link DirectConversationService}.
 * Contains business rules for participants, permissions, read receipts, and stats.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DirectConversationServiceImpl implements DirectConversationService {

    private final ConversationHelper conversationHelper;
    private final ConversationRepository conversationRepository;
    private final ConversationParticipantRepository participantRepository;

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public OutputConversationDto createOrGetDirectConversation(@NonNull UUID user1Id, @NonNull UUID user2Id) {
        Objects.requireNonNull(user1Id, "user1Id is null");
        Objects.requireNonNull(user2Id, "user2Id is null");
        try {
            if (user1Id.equals(user2Id)) {
                throw new WP2ValidationException("Cannot create direct conversation with the same user");
            }
            Optional<Conversation> existing = conversationRepository.findDirectConversationBetweenUsers(user1Id, user2Id);
            if (existing.isPresent()) {
                log.info("Found existing direct conversation between {} and {}: {}",
                        user1Id, user2Id, existing.get().getId());
                return mapToDirectOutputConversationDto(user1Id, existing.get());
            }

            UUID conversationId = UUID.randomUUID();
            Conversation conversation = Conversation.createDirectConversation(conversationId, user1Id, user2Id);
            conversation = conversationRepository.save(conversation);
            log.info("Created direct conversation entity: {} for users - {} and {}", conversationId, user1Id, user2Id);
            ConversationParticipant participant1 = ConversationParticipant.createMember(conversationId, user1Id);
            ConversationParticipant participant2 = ConversationParticipant.createMember(conversationId, user2Id);
            participantRepository.save(participant1);
            participantRepository.save(participant2);
            // user1 should already be saved by the session management service
            // so we only need to save user2 for direct conversations
            conversationHelper.saveConversationUsers(Set.of(user2Id));
            return mapToDirectOutputConversationDto(user1Id, conversation);
        } catch (Exception e) {
            log.error("Failed to create or get direct conversation between {} and {}: {}",
                    user1Id, user2Id, e.getMessage(), e);
            throw new RuntimeException("Failed to create direct conversation: " + e.getMessage(), e);
        }
    }

    private OutputConversationDto mapToDirectOutputConversationDto(UUID userId, Conversation conversation) {
        return conversationHelper.populateDetailsForDirect(userId, conversation);
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public OutputConversationDto getDirectConversation(@NonNull UUID user1Id, @NonNull UUID user2Id) {
        Objects.requireNonNull(user1Id, "user1Id is null");
        Objects.requireNonNull(user2Id, "user2Id is null");

        Optional<Conversation> conversationOpt =  conversationRepository.findDirectConversationBetweenUsers(user1Id, user2Id);

        if (conversationOpt.isEmpty()) {
            throw new WP2ResourceNotFoundException("conversation between %s and %s".formatted(user1Id, user2Id));
        }

        return mapToDirectOutputConversationDto(user1Id, conversationOpt.get());
    }
}
