package com.finconsgroup.itserr.marketplace.usercommunication.dm.service.impl;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.User;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.UserRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.security.WebSocketUser;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.SessionManagementService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Default implementation for {@link SessionManagementService}
 */
@ConditionalOnProperty(prefix = "user-communication.dm.web-socket", value = "enable-broker-relay", havingValue = "false",
        matchIfMissing = true)
@Service
@Slf4j
public class DefaultSessionManagementService extends AbstractSessionManagementService {

    public DefaultSessionManagementService(final UserRepository userRepository) {
        super(userRepository);
    }

    // Track active sessions per user
    private final Map<UUID, Set<String>> sessionIdsByUserId = new ConcurrentHashMap<>();
    // Track session to user mapping
    private final Map<String, WebSocketUser> userBySessionId = new ConcurrentHashMap<>();

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public User onSessionCreated(@NonNull String sessionId, @NonNull WebSocketUser user) {
        UUID userId = user.getUserId();

        boolean isFirstSession = !sessionIdsByUserId.containsKey(userId);
        AtomicInteger sessionCount = new AtomicInteger(0);

        // Track this session for the user
        userBySessionId.put(sessionId, user);

        // Update user session count
        sessionIdsByUserId.merge(userId, new HashSet<>(List.of(sessionId)), (s1, s2) -> {
            s1.addAll(s2);
            sessionCount.set(s1.size());
            return s1;
        });

        // If this is the user's first session, mark them as online
        if (isFirstSession) {
            log.info("User {} came online with session {}", userId, sessionId);
            return updateUserOnlineStatus(user, true);
        } else {
            log.info("User {} (session: {}, total sessions: {})", userId, sessionId, sessionCount.get());
            return null;
        }
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public User onSessionDestroyed(@NonNull String sessionId) {
        WebSocketUser user = userBySessionId.remove(sessionId);

        if (user != null) {
            UUID userId = user.getUserId();
            AtomicBoolean isLastSession = new AtomicBoolean(false);
            AtomicInteger sessionCount = new AtomicInteger(0);
            // Decrease user session count
            sessionIdsByUserId.computeIfPresent(userId, (key, sessions) -> {
                sessions.remove(sessionId);
                isLastSession.set(sessions.isEmpty());
                sessionCount.set(sessions.size());
                return sessions;
            });
            if (isLastSession.get()) {
                // This was the user's last session, mark them as offline
                sessionIdsByUserId.remove(userId);
                log.info("User {} went offline (session: {})", userId, sessionId);
                return updateUserOnlineStatus(user, false);
            } else {
                log.info("User {} disconnected session {} (remaining sessions: {})", userId, sessionId, sessionCount.get());
            }
        }
        return null;
    }

    @Override
    public boolean isUserOnline(@NonNull UUID userId) {
        return getUserSessionCount(userId) > 0;
    }

    @Override
    public int getUserSessionCount(@NonNull UUID userId) {
        return Optional.ofNullable(sessionIdsByUserId.get(userId)).map(Set::size).orElse(0);
    }

}
