package com.finconsgroup.itserr.marketplace.usercommunication.dm.service.impl;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.component.ConversationHelper;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessageDestinations;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputChatMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputChatMessageReadReceiptDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationMessageSummaryDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputUserDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ChatMessage;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.Conversation;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.MessageReadReceipt;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.ConversationType;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.mapper.ChatMessageMapper;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.mapper.MessageReadReceiptMapper;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.ChatMessageRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.ConversationRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.repository.MessageReadReceiptRepository;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.ChatMessageService;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.ChatRoomService;
import jakarta.persistence.EntityManager;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.messaging.simp.SimpMessagingTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.SecurityConstants.SYSTEM_USER_ID;

/**
 * Consumer component that processes messages from Messaging Service and persists/broadcasts them.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class ChatMessageServiceImpl implements ChatMessageService {

    private final ChatMessageRepository chatMessageRepository;
    private final ChatRoomService chatRoomService;
    private final ChatMessageMapper chatMessageMapper;
    private final MessageReadReceiptRepository messageReadReceiptRepository;
    private final MessageReadReceiptMapper messageReadReceiptMapper;
    private final ConversationRepository conversationRepository;
    private final ConversationHelper conversationHelper;
    private final EntityManager entityManager;
    private final SimpMessagingTemplate messagingTemplate;
    private final MessageDestinations messageDestinations;

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public void handleChatMessage(ChatMessage message) {
        try {
            // Process the message based on its type
            switch (message.getMessageType()) {
                case TEXT:
                    handleTextMessage(message);
                    break;
                // TODO: Check Later as this feature is not a priority for now
//                case JOIN:
//                    handleJoinMessage(message);
//                    break;
//                case LEAVE:
//                    handleLeaveMessage(message);
//                    break;
                case TYPING:
                    handleTypingMessage(message);
                    break;
                case SYSTEM:
                    handleSystemMessage(message);
                    break;
                case FILE:
                case IMAGE:
                    handleMediaMessage(message);
                    break;
                default:
                    log.warn("Unknown message type: {} for message: {}",
                            message.getMessageType(), message.getId());
            }

            log.info("Chat message processed successfully: {}", message.getId());

        } catch (Exception e) {
            log.error("Failed to process chat message: {}", message.getId(), e);
            // In a real application, you might want to send the message to a dead letter queue
        }
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public void handleUserNotification(ChatMessage notification) {
        try {
            // Store notification in the database
            chatMessageRepository.save(notification);

            log.info("User notification processed successfully: {}", notification.getId());

        } catch (Exception e) {
            log.error("Failed to process user notification: {}", notification.getId(), e);
        }
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public List<ChatMessage> getConversationMessages(UUID conversationId) {
        return chatMessageRepository.findByConversationIdOrderByCreatedAtAsc(conversationId);
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public Page<OutputChatMessageDto> getConversationMessages(Conversation conversation, Pageable pageable) {
        Page<ChatMessage> chatMessagesPage = chatMessageRepository.findMessagesByConversationId(
                conversation.getId(), pageable);
        List<OutputChatMessageDto> chatMessageDtos = mapToOutputChatMessageDtos(chatMessagesPage.getContent(),
                List.of(conversation));
        return new PageImpl<>(chatMessageDtos, chatMessagesPage.getPageable(), chatMessagesPage.getTotalElements());
    }

    @Override
    public int markConversationAsRead(Conversation conversation, UUID userId, Instant readAt) {
        if (conversation.isDirect()) {
            List<UUID> markedMessageIds = chatMessageRepository.getConversationMessagesToMarkAsRead(conversation.getId(),
                    userId, readAt);
            markDirectMessagesAsRead(conversation, readAt, markedMessageIds);
            return markedMessageIds.size();
        } else {
            List<UUID> markedMessageIds = messageReadReceiptRepository.getAllMessagesToMarkAsReadByUser(userId,
                    conversation.getId());
            markGroupMessagesAsRead(conversation, userId, readAt, markedMessageIds);
            return markedMessageIds.size();
        }
    }

    @Override
    public int markConversationMessagesUptoCreatedAtAsRead(Conversation conversation, UUID userId, Instant readAt,
                                                           Instant uptoCreatedAt) {
        if (conversation.isDirect()) {
            List<UUID> markedMessageIds = chatMessageRepository.getConversationMessagesUptoCreatedAtToMarkAsRead(conversation.getId(),
                    uptoCreatedAt, userId, readAt);
            markDirectMessagesAsRead(conversation, readAt, markedMessageIds);
            return markedMessageIds.size();
        } else {
            List<UUID> markedMessageIds = messageReadReceiptRepository.getMessagesUptoCreatedAtToMarkAsReadByUser(userId,
                    conversation.getId(), uptoCreatedAt);
            markGroupMessagesAsRead(conversation, userId, readAt, markedMessageIds);
            return markedMessageIds.size();
        }
    }

    @Override
    public int markConversationBulkMessagesAsRead(Conversation conversation, UUID userId, Instant readAt,
                                                  Set<UUID> messageIds) {
        if (conversation.isDirect()) {
            List<UUID> markedMessageIds = chatMessageRepository.getConversationMessagesByIdInToMarkAsRead(conversation.getId(),
                    messageIds, userId, readAt);
            markDirectMessagesAsRead(conversation, readAt, markedMessageIds);
            return markedMessageIds.size();
        } else {
            List<UUID> markedMessageIds = messageReadReceiptRepository.getMessagesByIdInToMarkAsReadByUser(userId,
                    conversation.getId(), messageIds);
            markGroupMessagesAsRead(conversation, userId, readAt, markedMessageIds);
            return markedMessageIds.size();
        }
    }

    private void markDirectMessagesAsRead(Conversation conversation, Instant readAt, List<UUID> messageIds) {
        if (messageIds == null || messageIds.isEmpty()) {
            return;
        }
        try {
            final int messageCount = messageIds.size();
            final int batchSize = 20;
            int offset = 0;
            int batchCount = Math.ceilDiv(messageCount, batchSize);

            for (int i = 0; i < batchCount; i++) {
                List<UUID> batchMessageIds = messageIds.subList(offset, Math.min(offset + batchSize, messageCount));
                List<ChatMessage> chatMessages = chatMessageRepository.findMessagesByConversationIdAndIdIn(conversation.getId(),
                        batchMessageIds);
                List<ChatMessage> savedChatMessages = new ArrayList<>(messageCount);
                for (ChatMessage chatMessage : chatMessages) {
                    chatMessage.setReadByReceiver(true);
                    chatMessage.setReadAt(readAt);
                    savedChatMessages.add(chatMessageRepository.saveAndFlush(chatMessage));
                }

                // send the read receipts to conversation topic
                List<OutputChatMessageDto> chatMessageDtos = mapToOutputChatMessageDtos(savedChatMessages, List.of(conversation));
                for (OutputChatMessageDto chatMessageDto : chatMessageDtos) {
                    messagingTemplate.convertAndSend(messageDestinations.getConversationTopic(conversation.getId()),
                            chatMessageDto);
                }
            }
        } catch (Exception e) {
            log.warn("Failed to send direct message read receipts for message ids - {}", e.getMessage(), e);
        }
    }

    private void markGroupMessagesAsRead(Conversation conversation, UUID userId, Instant readAt, List<UUID> messageIds) {
        if (messageIds == null || messageIds.isEmpty()) {
            return;
        }
        try {
            final int messageCount = messageIds.size();
            final int batchSize = 20;
            int offset = 0;
            int batchCount = Math.ceilDiv(messageCount, batchSize);

            for (int i = 0; i < batchCount; i++) {
                List<UUID> batchMessageIds = messageIds.subList(offset, Math.min(offset + batchSize, messageCount));
                List<ChatMessage> chatMessages = chatMessageRepository.findMessagesByConversationIdAndIdIn(conversation.getId(),
                        batchMessageIds);
                for (ChatMessage chatMessage : chatMessages) {
                    MessageReadReceipt messageReadReceipt = new MessageReadReceipt();
                    messageReadReceipt.setConversationId(conversation.getId());
                    messageReadReceipt.setMessageId(chatMessage.getId());
                    messageReadReceipt.setUserId(userId);
                    messageReadReceipt.setCreatedAt(readAt);
                    messageReadReceipt.setReadAt(readAt);
                    messageReadReceiptRepository.saveAndFlush(messageReadReceipt);
                }

                // send the read receipts to conversation topic
                List<OutputChatMessageDto> chatMessageDtos = mapToOutputChatMessageDtos(chatMessages, List.of(conversation));
                for (OutputChatMessageDto chatMessageDto : chatMessageDtos) {
                    messagingTemplate.convertAndSend(messageDestinations.getConversationTopic(conversation.getId()),
                            chatMessageDto);
                }
            }
        } catch (Exception e) {
            log.warn("Failed to send group message read receipts for message ids - {}", e.getMessage(), e);
        }
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public Map<UUID, OutputConversationMessageSummaryDto> getConversationMessageSummary(UUID userId,
                                                                                        List<Conversation> conversations) {
        if (conversations == null || conversations.isEmpty()) {
            return Map.of();
        }

        Set<UUID> conversationIds = new HashSet<>();
        Set<UUID> directConversationIds = new HashSet<>();
        Set<UUID> groupConversationIds = new HashSet<>();
        conversations.forEach(conversation -> {
            conversationIds.add(conversation.getId());
            if (conversation.isDirect()) {
                directConversationIds.add(conversation.getId());
            } else if (conversation.isGroup()) {
                groupConversationIds.add(conversation.getId());
            }
        });

        Map<UUID, OutputConversationMessageSummaryDto> messageSummaryByConversationId =
                chatMessageRepository.countByConversationId(conversationIds).stream().collect(
                        Collectors.toMap(OutputConversationMessageSummaryDto::getConversationId, Function.identity()));
        List<ChatMessage> latestChatMessages = chatMessageRepository.findLatestByConversationIdIn(conversationIds);
        List<ChatMessage> earliestUnreadChatMessages = new LinkedList<>();
        Map<UUID, Long> unreadCountByConversationId = new HashMap<>();
        if (!directConversationIds.isEmpty()) {
            unreadCountByConversationId.putAll(chatMessageRepository.countUnreadByConversationIdIn(userId, directConversationIds)
                    .stream().collect(Collectors.toMap(OutputConversationMessageSummaryDto::getConversationId,
                            OutputConversationMessageSummaryDto::getMessageCount)));
            earliestUnreadChatMessages.addAll(chatMessageRepository.findEarliestUnreadByConversationIdIn(userId, directConversationIds));
        }
        if (!groupConversationIds.isEmpty()) {
            unreadCountByConversationId.putAll(messageReadReceiptRepository.countUnreadByConversationIdIn(userId, groupConversationIds)
                    .stream().collect(Collectors.toMap(OutputConversationMessageSummaryDto::getConversationId,
                            OutputConversationMessageSummaryDto::getMessageCount)));
            earliestUnreadChatMessages.addAll(messageReadReceiptRepository.findEarliestUnreadByConversationIdIn(userId, groupConversationIds));
        }

        Map<UUID, ChatMessage> chatMessageById = new HashMap<>();
        Map<UUID, UUID> latestChatMessageByConversationId = new HashMap<>();
        Map<UUID, UUID> earliestUnreadChatMessageByConversationId = new HashMap<>();
        latestChatMessages.forEach(chatMessage -> {
            chatMessageById.put(chatMessage.getId(), chatMessage);
            latestChatMessageByConversationId.put(chatMessage.getConversationId(), chatMessage.getId());
        });
        earliestUnreadChatMessages.forEach(chatMessage -> {
            chatMessageById.put(chatMessage.getId(), chatMessage);
            earliestUnreadChatMessageByConversationId.put(chatMessage.getConversationId(), chatMessage.getId());
        });

        List<OutputChatMessageDto> chatMessageDtos = mapToOutputChatMessageDtos(List.copyOf(chatMessageById.values()),
                conversations);

        Map<UUID, OutputChatMessageDto> chatMessageDtoById = chatMessageDtos.stream()
                .collect(Collectors.toMap(
                        OutputChatMessageDto::getId,
                        Function.identity()
                ));

        return conversationIds.stream().collect(Collectors.toMap(
                Function.identity(),
                conversationId -> {
                    OutputConversationMessageSummaryDto messageSummaryDto = messageSummaryByConversationId.get(conversationId);
                    if (messageSummaryDto == null) {
                        return new OutputConversationMessageSummaryDto(conversationId, 0L);
                    }

                    messageSummaryDto.setUnreadMessageCount(unreadCountByConversationId.getOrDefault(conversationId, 0L));
                    UUID latestMessageId = latestChatMessageByConversationId.get(conversationId);
                    if (latestMessageId != null) {
                        messageSummaryDto.setLatestMessage(chatMessageDtoById.get(latestMessageId));
                    }
                    UUID earliestMessageId = earliestUnreadChatMessageByConversationId.get(conversationId);
                    if (earliestMessageId != null) {
                        messageSummaryDto.setEarliestUnreadMessage(chatMessageDtoById.get(earliestMessageId));
                    }
                    return messageSummaryDto;
                }
        ));
    }

    private List<OutputChatMessageDto> mapToOutputChatMessageDtos(@NonNull List<ChatMessage> chatMessages,
                                                                  @NonNull List<Conversation> conversations) {
        Set<UUID> chatUserIds = chatMessages.stream().map(ChatMessage::getSenderId).collect(Collectors.toSet());
        Map<UUID, OutputUserDto> userDtoById = conversationHelper.mapUserIdsToDtos(chatUserIds, true);
        Map<UUID, Conversation> conversationDtoById = conversations
                .stream().collect(Collectors.toMap(Conversation::getId, Function.identity()));
        Set<UUID> groupChatMessageIds = new HashSet<>();
        List<OutputChatMessageDto> chatMessageDtos = new LinkedList<>();
        chatMessages.forEach(chatMessage -> {
            OutputChatMessageDto chatMessageDto = chatMessageMapper.entityToOutputChatMessageDto(chatMessage);
            chatMessageDto.setSender(userDtoById.get(chatMessage.getSenderId()));
            Conversation conversation = conversationDtoById.get(chatMessage.getConversationId());
            if (conversation.getConversationType() == ConversationType.GROUP) {
                groupChatMessageIds.add(chatMessage.getId());
            } else if (conversation.getConversationType() == ConversationType.DIRECT) {
                // create a dummy read receipt for direct conversations
                if (chatMessage.getReadAt() != null) {
                    chatMessageDto.setReadReceipts(List.of(
                                    OutputChatMessageReadReceiptDto.builder()
                                            .id(chatMessage.getId()) // use the id of the chat message itself
                                            .userId(chatMessage.getReceiverId())
                                            .readAt(chatMessage.getReadAt())
                                            .createdAt(chatMessage.getReadAt())
                                            .build()
                            )
                    );
                }
            }
            chatMessageDtos.add(chatMessageDto);
        });

        populateGroupReadReceipts(chatMessageDtos, groupChatMessageIds);
        return chatMessageDtos;
    }

    private void populateGroupReadReceipts(List<OutputChatMessageDto> chatMessageDtos,
                                           Set<UUID> groupChatMessageIds) {
        if (groupChatMessageIds.isEmpty()) {
            return;
        }

        Map<UUID, List<MessageReadReceipt>> messageReadReceiptsByMessageId = messageReadReceiptRepository.findByMessageIdIn(groupChatMessageIds)
                .stream()
                .collect(Collectors.groupingBy(
                        mrr -> mrr.getMessage().getId(),
                        Collectors.mapping(Function.identity(), Collectors.toList())
                ));

        chatMessageDtos.forEach(chatMessageDto ->
                chatMessageDto.setReadReceipts(
                        Optional.ofNullable(messageReadReceiptsByMessageId.get(chatMessageDto.getId()))
                                .map(mrr -> mrr.stream()
                                        .map(messageReadReceiptMapper::entityToOutputChatMessageReadReceiptDto)
                                        .collect(Collectors.toList()))
                                .orElse(null))
        );
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public List<ChatMessage> getUserNotifications(UUID userId) {
        return chatMessageRepository.findBySenderIdOrderByCreatedAtDesc(SYSTEM_USER_ID)
                .stream()
                .filter(msg -> userId.equals(msg.getReceiverId()))
                .toList();
    }

    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    @Override
    public void clearUserNotifications(UUID userId) {
        @SuppressWarnings("squid:S6809")
        List<ChatMessage> notifications = getUserNotifications(userId);
        chatMessageRepository.deleteAll(notifications);
        log.info("Cleared {} notifications for user: {}", notifications.size(), userId);
    }

    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    @Override
    public Page<OutputChatMessageDto> searchMessages(UUID userId, String searchTerm, Pageable pageable) {
        Page<OutputChatMessageDto> chatMessageDtoPage =
                chatMessageRepository.findByContentContainingIgnoreCase(userId, searchTerm, pageable)
                        .map(chatMessageMapper::entityToOutputChatMessageDto);

        Set<UUID> chatMessageIds = new HashSet<>();
        Set<UUID> conversationIds = new HashSet<>();
        Set<UUID> chatUserIds = new HashSet<>();
        chatMessageDtoPage.forEach(chatMessageDto -> {
            chatMessageIds.add(chatMessageDto.getId());
            conversationIds.add(chatMessageDto.getConversationId());
            chatUserIds.add(chatMessageDto.getSenderId());
        });

        if (!chatMessageIds.isEmpty()) {
            Map<UUID, OutputConversationDto> conversationDtoById =
                    conversationHelper.loadConversations(userId, conversationIds)
                            .stream().collect(Collectors.toMap(OutputConversationDto::getId, Function.identity()));

            Map<UUID, OutputUserDto> userDtoById = conversationHelper.mapUserIdsToDtos(chatUserIds, true);

            chatMessageDtoPage.getContent().forEach(chatMessageDto -> {
                OutputConversationDto conversationDto = conversationDtoById.get(chatMessageDto.getConversationId());
                if (conversationDto != null) {
                    chatMessageDto.setConversationName(conversationDto.getName());
                }
                chatMessageDto.setSender(userDtoById.get(chatMessageDto.getSenderId()));
            });

        }

        return chatMessageDtoPage;
    }

    @Override
    public OutputChatMessageDto getOutputChatMessageDto(@NonNull ChatMessage chatMessage, boolean loadUserProfiles) {
        OutputChatMessageDto chatMessageDto = chatMessageMapper.entityToOutputChatMessageDto(chatMessage);
        if (!SYSTEM_USER_ID.equals(chatMessage.getSenderId())) {
            OutputUserDto sender = conversationHelper.mapUserIdsToDtos(Set.of(chatMessage.getSenderId()), loadUserProfiles)
                    .get(chatMessageDto.getSenderId());
            chatMessageDto.setSender(sender);
        }
        return chatMessageDto;
    }

    private void handleTextMessage(ChatMessage message) {
        // Store the message in the database
        UUID conversationId = message.getConversationId();
        if (conversationId != null) {
            saveChatMessage(message);

            // Update conversation activity
            conversationRepository.updateLastActivity(conversationId, Instant.now());

            // Note: Message broadcasting to conversation topic is handled by WebSocketChatController @SendTo annotation
            // Removed duplicate broadcast to prevent message triplication
        }

        log.debug("Text message stored and marked as delivered for conversation: {}", conversationId);
    }

    // TODO: Check Later as this feature is not a priority for now
//    private void handleJoinMessage(ChatMessage message) {
//        UUID conversationId = message.getConversationId();
//        UUID userId = message.getSenderId();
//
//        if (conversationId != null && userId != null) {
//            // Add user to chat room
//            chatRoomService.addUserToChatRoom(conversationId, userId);
//
//            // Store the join message in database
//            chatMessageRepository.save(message);
//        }
//
//        log.info("User {} joined conversation: {}", userId, conversationId);
//    }
//
//    private void handleLeaveMessage(ChatMessage message) {
//        UUID conversationId = message.getConversationId();
//        UUID userId = message.getSenderId();
//
//        if (conversationId != null && userId != null) {
//            // Remove user from chat room
//            chatRoomService.removeUserFromChatRoom(conversationId, userId);
//
//            // Store the leave message in database
//            chatMessageRepository.save(message);
//        }
//
//        log.info("User {} left conversation: {}", userId, conversationId);
//    }

    private void handleTypingMessage(ChatMessage message) {
        // Typing indicators are typically not stored permanently
        // They are just broadcasted to other users in the conversation
        log.debug("Typing indicator from user: {} in conversation: {}",
                message.getSenderId(), message.getConversationId());
    }

    private void handleSystemMessage(ChatMessage message) {
        UUID conversationId = message.getConversationId();
        if (conversationId != null) {
            saveChatMessage(message);
            chatMessageRepository.findById(message.getId())
                    .ifPresent(chatMessage -> message.setMessageIndex(chatMessage.getMessageIndex()));

            // Note: Message broadcasting to conversation topic is handled by WebSocketChatController @SendTo annotation
            // Removed duplicate broadcast to prevent message triplication
        }

        log.info("System message processed and marked as delivered for conversation: {}", conversationId);
    }

    private void handleMediaMessage(ChatMessage message) {
        // Handle file and image messages
        UUID conversationId = message.getConversationId();
        if (conversationId != null) {
            saveChatMessage(message);
            conversationRepository.updateLastActivity(conversationId, Instant.now());

            // Note: Message broadcasting to conversation topic is handled by WebSocketChatController @SendTo annotation
            // Removed duplicate broadcast to prevent message triplication
        }

        log.info("Media message processed and marked as delivered: {} in conversation: {}",
                message.getMessageType(), conversationId);
    }

    private void saveChatMessage(ChatMessage message) {
        // Mark message as delivered when successfully processed
        message.setDelivered(true);
        ChatMessage savedMessage = chatMessageRepository.saveAndFlush(message);
        // set the message index for newly saved message
        int updateCount = chatMessageRepository.updateMessageIndex(message.getConversationId());
        log.debug("message index updated for row: {}", updateCount);
        // refresh the chat message entity if any updates were applied
        if (updateCount > 0) {
            entityManager.refresh(savedMessage);
            message.setMessageIndex(savedMessage.getMessageIndex());
        }
    }
}