package com.finconsgroup.itserr.marketplace.usercommunication.dm.service;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputUnreadMessageSummaryDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ChatMessage;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.OfflineMessage;

import java.util.List;
import java.util.UUID;

/**
 * Service contract for handling offline message queuing, delivery, and notifications.
 */
public interface OfflineMessageService {
    /**
     * Queue a message for a recipient who is currently offline (or when direct delivery fails).
     * @param chatMessage the message to queue
     * @param recipientUserId the offline recipient's user ID
     */
    void queueMessageForOfflineUser(ChatMessage chatMessage, UUID recipientUserId);

    /**
     * Notify a user that they have pending messages, typically when they connect/subscribe.
     * @param userId the user to notify
     */
    void notifyUserAboutPendingMessages(UUID userId);

    /**
     * Deliver all queued messages to a user and mark them delivered.
     * @param userId the recipient user ID
     */
    void deliverOfflineMessages(UUID userId);

    /**
     * Best-effort check if a user is online (e.g., via WebSocket reachability).
     * @param userId the user ID
     * @return true if the user appears online
     */
    boolean isUserOnline(UUID userId);

    /**
     * Count undelivered messages for a user across all conversations.
     * @param userId the user ID
     * @return number of undelivered messages
     */
    long getUndeliveredMessageCount(UUID userId);

    /**
     * Count undelivered messages for a user in a specific conversation.
     * @param userId the user ID
     * @param conversationId the conversation ID
     * @return number of undelivered messages in the conversation
     */
    long getUndeliveredMessageCount(UUID userId, UUID conversationId);

    /**
     * Retrieve undelivered messages for a user (oldest first).
     * @param userId the user ID
     * @return list of offline message records
     */
    List<OfflineMessage> getUndeliveredMessages(UUID userId);

    /**
     * Get unread message summary for a user
     *
     * @param userId the user ID
     * @return the unread message summary
     */
    OutputUnreadMessageSummaryDto getUnreadMessageSummary(UUID userId);

    /**
     * Cleanup delivered messages older than a threshold to keep storage tidy.
     * @param daysOld threshold in days
     */
    void cleanupOldDeliveredMessages(int daysOld);

    /**
     * Attempt immediate delivery if user online; otherwise queue for later.
     * @param chatMessage the message
     * @param recipientUserId the recipient user ID
     */
    void handleMessageDelivery(ChatMessage chatMessage, UUID recipientUserId);

    /**
     * Send a generic offline notification to a user about pending messages.
     * Kept for backward compatibility; notifyUserAboutPendingMessages is preferred.
     * @param userId the user ID
     */
    void sendOfflineMessageNotification(UUID userId);
}