package com.finconsgroup.itserr.marketplace.usercommunication.dm.security;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessageDestinations;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.PreConditionService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageChannel;
import org.springframework.messaging.simp.stomp.StompCommand;
import org.springframework.messaging.simp.stomp.StompHeaderAccessor;
import org.springframework.messaging.support.ChannelInterceptor;
import org.springframework.messaging.support.MessageHeaderAccessor;

import java.util.UUID;

/**
 * An implementation of the {@link ChannelInterceptor} to verify if the users are allowed to subscribe
 * to the topics.
 */
@RequiredArgsConstructor
@Slf4j
public final class SubscriptionAuthorizationChannelInterceptor implements ChannelInterceptor {

    private static final String INVALID_CONVERSATION_ID = "<NOT_VALID>";

    private final PreConditionService preConditionService;
    private final MessageDestinations messageDestinations;

    @Override
    public Message<?> preSend(@NonNull Message<?> message, @NonNull MessageChannel channel) {
        StompHeaderAccessor accessor = MessageHeaderAccessor.getAccessor(message, StompHeaderAccessor.class);
        if (accessor != null && StompCommand.SUBSCRIBE.equals(accessor.getCommand())) {
            // the authentication interceptor should already set the user header to WebSocketAuthentication object
            WebSocketAuthentication authentication = (WebSocketAuthentication) accessor.getUser();
            UUID sessionUserId = authentication.getPrincipal().getUserId();
            String destination = accessor.getDestination();
            if (destination != null) {
                if (!checkTopicConversationDestination(sessionUserId, destination)) {
                    return null;
                }
            }

        }
        return message;
    }

    /*
     * Checks if user has access to the conversation, if not then the subscription is not added
     */
    private boolean checkTopicConversationDestination(UUID sessionUserId, String destination) {
        UUID conversationId = null;
        if (messageDestinations.isTopicConversation(destination)) {
            try {
                conversationId = messageDestinations.extractConversationId(destination);
                preConditionService.checkUserAndConversation(sessionUserId, conversationId);
            } catch (Exception e) {
                log.warn("User {} attempted to subscribe to destination {} for conversation {} and failed with {}",
                        sessionUserId, destination, conversationId == null ? INVALID_CONVERSATION_ID : conversationId,
                        e.getMessage());
                return false;
            }
        }
        return true;
    }
}
