package com.finconsgroup.itserr.marketplace.usercommunication.dm.repository;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ConversationParticipant;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.ParticipantRole;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

@Repository
public interface ConversationParticipantRepository extends JpaRepository<ConversationParticipant, UUID> {

    /**
     * Find a participant by conversation ID and user ID
     */
    Optional<ConversationParticipant> findByConversationIdAndUserId(UUID conversationId, UUID userId);

    /**
     * Find all active participants in a conversation
     */
    List<ConversationParticipant> findByConversationIdAndActiveIsTrue(UUID conversationId);

    /**
     * Find all conversations for a user (active participation)
     */
    List<ConversationParticipant> findByUserIdAndActiveIsTrue(UUID userId);

    /**
     * Check if a user is a participant in a conversation
     */
    boolean existsByConversationIdAndUserIdAndActiveIsTrue(UUID conversationId, UUID userId);

    /**
     * Check if a user has admin privileges in a conversation
     */
    @Query("SELECT CASE WHEN COUNT(cp) > 0 THEN true ELSE false END FROM ConversationParticipant cp WHERE " +
           "cp.conversationId = :conversationId AND cp.userId = :userId AND cp.active = true AND " +
           "(cp.role = 'OWNER' OR cp.role = 'ADMIN')")
    boolean hasAdminPrivileges(@Param("conversationId") UUID conversationId, @Param("userId") UUID userId);

    /**
     * Count active participants in a conversation
     */
    long countByConversationIdAndActiveIsTrue(UUID conversationId);

    /**
     * Update participant role
     */
    @Modifying
    @Query("UPDATE ConversationParticipant cp SET cp.role = :role WHERE cp.conversationId = :conversationId AND cp.userId = :userId")
    void updateParticipantRole(@Param("conversationId") UUID conversationId, 
                              @Param("userId") UUID userId, 
                              @Param("role") ParticipantRole role);

    /**
     * Update participant active status
     */
    @Modifying
    @Query("UPDATE ConversationParticipant cp SET cp.active = :isActive WHERE cp.conversationId = :conversationId AND cp.userId = :userId")
    void updateParticipantActiveStatus(@Param("conversationId") UUID conversationId, 
                                      @Param("userId") UUID userId, 
                                      @Param("isActive") boolean isActive);

    /**
     * Update participant mute status
     */
    @Modifying
    @Query("UPDATE ConversationParticipant cp SET cp.muted = :isMuted WHERE cp.conversationId = :conversationId AND cp.userId = :userId")
    void updateParticipantMuteStatus(@Param("conversationId") UUID conversationId, 
                                    @Param("userId") UUID userId, 
                                    @Param("isMuted") boolean isMuted);

    /**
     * Update participant notification settings
     */
    @Modifying
    @Query("UPDATE ConversationParticipant cp SET cp.notificationEnabled = :enabled WHERE cp.conversationId = :conversationId AND cp.userId = :userId")
    void updateNotificationSettings(@Param("conversationId") UUID conversationId, 
                                   @Param("userId") UUID userId, 
                                   @Param("enabled") boolean enabled);

    /**
     * Update last read timestamp for a participant
     */
    @Modifying
    @Query("UPDATE ConversationParticipant cp SET cp.lastReadAt = :timestamp WHERE cp.conversationId = :conversationId AND cp.userId = :userId")
    void updateLastReadAt(@Param("conversationId") UUID conversationId, 
                         @Param("userId") UUID userId, 
                         @Param("timestamp") Instant timestamp);

    /**
     * Remove participant from conversation (set inactive)
     */
    @Modifying
    @Query("UPDATE ConversationParticipant cp SET cp.active = false WHERE cp.conversationId = :conversationId AND cp.userId = :userId")
    void removeParticipant(@Param("conversationId") UUID conversationId, @Param("userId") UUID userId);

    /**
     * Find conversations where user has unread messages
     */
    @Query("SELECT cp FROM ConversationParticipant cp WHERE cp.userId = :userId AND cp.active = true AND " +
           "(cp.lastReadAt IS NULL OR cp.lastReadAt < (SELECT MAX(cm.createdAt) FROM ChatMessage cm WHERE cm.conversationId = cp.conversationId))")
    List<ConversationParticipant> findConversationsWithUnreadMessages(@Param("userId") UUID userId);
}