package com.finconsgroup.itserr.marketplace.usercommunication.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.MessageType;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.EnumType;
import jakarta.persistence.Enumerated;
import jakarta.persistence.FetchType;
import jakarta.persistence.OneToMany;
import jakarta.persistence.Table;
import jakarta.validation.constraints.NotBlank;
import jakarta.validation.constraints.NotNull;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.ToString;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.util.List;
import java.util.UUID;

/**
 * Entity representing a chat message exchanged in direct or group conversations.
 * Includes delivery/read flags to support receipts and offline delivery.
 */
@Entity
@Table(name = "chat_messages")
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ChatMessage extends AbstractUUIDEntity {

    @Column(name = "sender_id", nullable = false)
    @NotNull(message = "Sender ID cannot be blank")
    private UUID senderId;

    @Column(name = "receiver_id")
    private UUID receiverId;

    @Column(name = "content", nullable = false, columnDefinition = "TEXT")
    @NotBlank(message = "Content cannot be blank")
    private String content;

    @Enumerated(EnumType.STRING)
    @Column(name = "message_type", nullable = false)
    @NotNull(message = "Message type cannot be null")
    private MessageType messageType;

    @Column(name = "created_at", nullable = false)
    private Instant createdAt;

    @Column(name = "conversation_id")
    private UUID conversationId;

    @Column(name = "delivered", nullable = false, columnDefinition = "BOOLEAN DEFAULT FALSE")
    private boolean delivered;

    @Column(name = "delivered_at")
    private Instant deliveredAt;

    @Column(name = "read_by_receiver", nullable = false, columnDefinition = "BOOLEAN DEFAULT FALSE")
    private boolean readByReceiver;

    @Column(name = "read_at")
    private Instant readAt;

    @Column(name = "message_index")
    private Long messageIndex;

    /**
     * The event conductors related to the event.
     */
    @OneToMany(fetch = FetchType.LAZY, cascade = CascadeType.REMOVE, orphanRemoval = true, mappedBy = "message")
    @ToString.Exclude
    private List<MessageReadReceipt> readReceipts;

    public void setDelivered(boolean delivered) {
        this.delivered = delivered;
        if (delivered && this.deliveredAt == null) {
            this.deliveredAt = Instant.now();
        }
    }

    public void setReadByReceiver(boolean readByReceiver) {
        this.readByReceiver = readByReceiver;
        if (readByReceiver && this.readAt == null) {
            this.readAt = Instant.now();
        }
    }

    @Override
    public void prePersist() {
        super.prePersist();
        if (this.createdAt == null) {
            this.createdAt = Instant.now();
        }
    }
}