package com.finconsgroup.itserr.marketplace.usercommunication.dm.controller;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.api.WebSocketChatApi;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessageDestinations;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputChatMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputTypingMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputChatMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ChatMessage;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.MessageType;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.security.WebSocketAuthentication;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.WebSocketService;
import jakarta.validation.Valid;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.messaging.handler.annotation.DestinationVariable;
import org.springframework.messaging.handler.annotation.MessageMapping;
import org.springframework.messaging.handler.annotation.Payload;
import org.springframework.messaging.simp.SimpMessageHeaderAccessor;
import org.springframework.messaging.simp.SimpMessagingTemplate;
import org.springframework.stereotype.Controller;

import java.time.Instant;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessageHeaders.USER_DESTINATION_DEFAULTS;
import static com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.SecurityConstants.SYSTEM_USER_ID;

/**
 * STOMP/WebSocket controller implementing WebSocketChatApi.
 * Handles messaging, join/leave, typing indicators, read receipts, and notifications over WS.
 */
@Controller
@RequiredArgsConstructor
@Slf4j
public class WebSocketChatController implements WebSocketChatApi {

    private final SimpMessagingTemplate messagingTemplate;
    private final WebSocketService webSocketService;
    private final MessageDestinations messageDestinations;

    /**
     * Handle chat messages sent to a specific conversation
     */
    @MessageMapping("/conversation/{conversationId}")
    @Override
    public void sendMessage(@DestinationVariable UUID conversationId,
                            @Valid @Payload InputChatMessageDto inputChatMessageDto,
                            WebSocketAuthentication authentication,
                            SimpMessageHeaderAccessor headerAccessor) {
        try {

            UUID userId = authentication.getPrincipal().getUserId();
            OutputChatMessageDto outputChatMessageDto = webSocketService.sendMessage(userId, conversationId, inputChatMessageDto);
            outputChatMessageDto.setRequestMessageId(headerAccessor.getFirstNativeHeader("requestMessageId"));
            messagingTemplate.convertAndSend(messageDestinations.getConversationTopic(conversationId),
                    outputChatMessageDto);

        } catch (Exception e) {
            log.error("Failed to process WebSocket message for conversation {}: {}", conversationId, e.getMessage(), e);
        }
    }

    /**
     * Handle user joining a conversation
     */
    @MessageMapping("/conversation/{conversationId}/join")
    @Override
    public void addUser(@DestinationVariable UUID conversationId,
                        @Payload InputChatMessageDto inputChatMessageDto,
                        WebSocketAuthentication authentication,
                        SimpMessageHeaderAccessor headerAccessor) {
        try {
            UUID userId = authentication.getPrincipal().getUserId();
            OutputChatMessageDto chatMessageDto = webSocketService.addUser(userId, conversationId, inputChatMessageDto);
            messagingTemplate.convertAndSend(messageDestinations.getConversationTopic(conversationId),
                    chatMessageDto);
        } catch (Exception e) {
            log.error("Failed to process join message for conversation {}: {}", conversationId, e.getMessage(), e);
        }
    }

    /**
     * Handle user leaving a conversation
     */
    @MessageMapping("/conversation/{conversationId}/leave")
    @Override
    public void removeUser(@DestinationVariable UUID conversationId,
                                           @Payload InputChatMessageDto inputChatMessageDto,
                                           WebSocketAuthentication authentication,
                                           SimpMessageHeaderAccessor headerAccessor) {
        try {
            UUID userId = authentication.getPrincipal().getUserId();
            OutputChatMessageDto chatMessageDto = webSocketService.removeUser(userId, conversationId, inputChatMessageDto);
            messagingTemplate.convertAndSend(messageDestinations.getConversationTopic(conversationId),
                    chatMessageDto);
        } catch (Exception e) {
            log.error("Failed to process leave message for conversation {}: {}", conversationId, e.getMessage(), e);
        }
    }

    /**
     * Handle typing indicators
     */
    @MessageMapping("/conversation/{conversationId}/typing")
    @Override
    public void sendTypingIndicator(@DestinationVariable UUID conversationId,
                                    @Payload InputTypingMessageDto inputTypingMessageDto,
                                    WebSocketAuthentication authentication) {
        try {
            UUID userId = authentication.getPrincipal().getUserId();
            OutputChatMessageDto chatMessageDto = webSocketService.sendTypingIndicator(userId, conversationId, inputTypingMessageDto);
            messagingTemplate.convertAndSend(messageDestinations.getConversationTypingTopic(conversationId),
                    chatMessageDto);
        } catch (Exception e) {
            log.error("Failed to send typing indicator for conversation {}: {}", conversationId, e.getMessage(), e);
        }
    }

    /**
     * Send direct message to a specific user (creates or uses existing direct conversation)
     */
    @MessageMapping("/conversation/direct/{receiverId}")
    @Override
    public void sendDirectMessage(@DestinationVariable UUID receiverId,
                                  @Payload InputChatMessageDto inputChatMessageDto,
                                  WebSocketAuthentication authentication) {
        try {
            UUID userId = authentication.getPrincipal().getUserId();
            OutputChatMessageDto outputChatMessageDto = webSocketService.sendDirectMessage(userId, receiverId, inputChatMessageDto);

            // Send message to the conversation topic for real-time delivery
            messagingTemplate.convertAndSend(messageDestinations.getConversationTopic(outputChatMessageDto.getConversationId()),
                    outputChatMessageDto);

            // Also send directly to the recipient for immediate notification
            messagingTemplate.convertAndSendToUser(
                    receiverId.toString(),
                    messageDestinations.getUserMessagesSuffix(),
                    outputChatMessageDto,
                    USER_DESTINATION_DEFAULTS
            );

            log.info("Direct message sent from {} to {} in conversation {}: {}",
                    outputChatMessageDto.getSenderId(), receiverId,
                    outputChatMessageDto.getConversationId(), outputChatMessageDto.getId());

        } catch (Exception e) {
            log.error("Failed to send direct message: {}", e.getMessage(), e);
        }
    }

    /**
     * Broadcast system message to all users in a conversation
     */
    @Override
    public void broadcastSystemMessage(UUID conversationId, String content) {
        try {
            OutputChatMessageDto outputChatMessageDto = webSocketService.broadcastSystemMessage(conversationId, content);

            if (outputChatMessageDto != null) {
                // Broadcast via WebSocket
                messagingTemplate.convertAndSend(messageDestinations.getConversationTopic(conversationId), outputChatMessageDto);

                log.info("System message broadcasted to conversation {}: {}", conversationId, content);
            }

        } catch (Exception e) {
            log.error("Failed to broadcast system message to conversation {}: {}", conversationId, e.getMessage(), e);
        }
    }

    /**
     * Send notification to a specific user
     */
    @Override
    public void sendUserNotification(UUID userId, String content, MessageType messageType) {
        try {

            OutputChatMessageDto outputChatMessageDto = webSocketService.sendUserNotification(userId, content, messageType);

            // Send via WebSocket for real-time delivery
            messagingTemplate.convertAndSendToUser(userId.toString(), messageDestinations.getUserNotificationsSuffix(),
                    outputChatMessageDto, USER_DESTINATION_DEFAULTS);

            log.info("Notification sent to user {}: {}", userId, content);

        } catch (Exception e) {
            log.error("Failed to send notification to user {}: {}", userId, e.getMessage(), e);
        }
    }

    /**
     * Send invitation notification for group chats
     */
    @Override
    public void sendInvitationNotification(UUID conversationId, UUID invitedUserId,
                                           WebSocketAuthentication authentication) {
        // TODO: Check if invitedBy should be set from principal instead
        try {
            UUID userId = authentication.getPrincipal().getUserId();
            OutputChatMessageDto outputChatMessageDto = webSocketService.sendInvitation(userId, conversationId, invitedUserId);

            if (outputChatMessageDto != null) {
                // Send via WebSocket for real-time delivery
                messagingTemplate.convertAndSendToUser(invitedUserId.toString(), messageDestinations.getUserInvitationsSuffix(),
                        outputChatMessageDto, USER_DESTINATION_DEFAULTS);

                log.info("Invitation notification sent to user {} for conversation {}", invitedUserId, conversationId);
            }

        } catch (Exception e) {
            log.error("Failed to send invitation notification: {}", e.getMessage(), e);
        }
    }

    /**
     * Broadcast message status update to conversation participants
     */
    public void broadcastMessageStatusUpdate(UUID conversationId, String messageId, boolean delivered, boolean read, String readByUserId) {
        try {
            ChatMessage statusUpdate = new ChatMessage();
            statusUpdate.setId(UUID.randomUUID());
            statusUpdate.setSenderId(SYSTEM_USER_ID);
            statusUpdate.setContent("MESSAGE_STATUS_UPDATE:" + messageId + ":" + delivered + ":" + read + ":" + (readByUserId != null ? readByUserId : ""));
            statusUpdate.setMessageType(MessageType.SYSTEM);
            statusUpdate.setConversationId(conversationId);
            statusUpdate.setCreatedAt(Instant.now());
            statusUpdate.setDelivered(delivered);
            statusUpdate.setReadByReceiver(read);

            // Send status update to all conversation participants
            messagingTemplate.convertAndSend(messageDestinations.getConversationChildTopic(conversationId, "status"), statusUpdate);

            log.debug("Message status update broadcasted for message {} in conversation {}: delivered={}, read={}",
                    messageId, conversationId, delivered, read);

        } catch (Exception e) {
            log.error("Failed to broadcast message status update for message {} in conversation {}: {}",
                    messageId, conversationId, e.getMessage(), e);
        }
    }
}