package com.finconsgroup.itserr.marketplace.usercommunication.dm.controller;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.api.ConversationApi;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputBulkMessagesDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputMuteConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputChatMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationParticipantDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationStatsDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputSearchResultDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputUnreadMessageSummaryDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.ConversationService;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.OfflineMessageService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.UUID;

/**
 * REST Controller for managing shared functionality for all conversations (DIRECT, GROUP)
 * Provides LinkedIn-style direct messaging and group chat functionality
 */
@RestController
@RequiredArgsConstructor
@Slf4j
public class ConversationController implements ConversationApi {

    private final ConversationService conversationService;
    private final OfflineMessageService offlineMessageService;

    /**
     * Get conversation by ID
     */
    @Override
    public OutputConversationDto getConversationById(UUID conversationId) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return conversationService.getConversation(userId, conversationId);
    }

    /**
     * Get recent conversations for a user (ordered by activity)
     */
    @Override
    public Page<OutputConversationDto> getRecentConversationsForUser(
            final int pageNumber,
            final int pageSize,
            final String sort,
            final Sort.Direction direction
    ) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        Pageable pageable = PageRequest.of(pageNumber, pageSize, Sort.by(direction, sort));
        return conversationService.getRecentConversationsForUser(userId, pageable);
    }

    /**
     * Get conversation participants
     */
    @Override
    public List<OutputConversationParticipantDto> getConversationParticipants(UUID conversationId) {
        return conversationService.getConversationParticipants(conversationId);
    }

    /**
     * Leave conversation
     */
    @Override
    public void leaveConversation(UUID conversationId) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        conversationService.leaveConversation(conversationId, userId);
    }

    /**
     * Mute/unmute conversation
     */
    @Override
    public void muteConversation(UUID conversationId, InputMuteConversationDto request) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        conversationService.muteConversation(conversationId, userId, request.isMuted());
    }

    @Override
    public void markConversationAsRead(UUID conversationId) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        conversationService.markConversationAsRead(userId, conversationId);
    }

    @Override
    public void markConversationUptoMessageAsRead(UUID conversationId, UUID messageId) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        conversationService.markConversationUptoMessageAsRead(userId, conversationId, messageId);
    }

    @Override
    public void markBulkMessagesAsRead(UUID conversationId, InputBulkMessagesDto request) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        conversationService.markBulkMessagesAsRead(userId, conversationId, request);
    }

    /**
     * Search conversations
     */
    @Override
    public OutputSearchResultDto searchConversations(final String query,
                                                                     final int pageNumber,
                                                                     final int pageSize,
                                                                     final String sort,
                                                                     final Sort.Direction direction) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        Pageable pageable = PageRequest.of(pageNumber, pageSize, Sort.by(direction, sort));
        return conversationService.searchConversations(userId, query, pageable);
    }

    /**
     * Get conversation statistics
     */
    @Override
    public OutputConversationStatsDto getConversationStats() {
        return conversationService.getConversationStats();
    }

    /**
     * Get conversation message history with pagination support for lazy loading
     */
    @Override
    public Page<OutputChatMessageDto> getConversationMessages(final UUID conversationId,
                                                                              final int pageNumber,
                                                                              final int pageSize,
                                                                              final String sort,
                                                                              final Sort.Direction direction) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        Pageable pageable = PageRequest.of(pageNumber, pageSize, Sort.by(direction, sort));
        return conversationService.getConversationMessages(userId,
                conversationId, pageable);
    }

    /**
     * Get unread message summary for a user
     */
    @Override
    public OutputUnreadMessageSummaryDto getUnreadMessageSummary() {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return offlineMessageService.getUnreadMessageSummary(userId);
    }

}