package com.finconsgroup.itserr.marketplace.usercommunication.dm.service;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputBulkMessagesDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputChatMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationParticipantDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationStatsDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputSearchResultDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ConversationParticipant;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;

import java.util.List;
import java.util.UUID;

/**
 * Service contract for managing conversations (direct and group).
 * Each method is documented to improve generated JavaDoc and maintainability.
 */
public interface ConversationService {

    /**
     * Get conversation details by ID.
     * @param userId the user id
     * @param conversationId the conversation ID
     * @return Output conversation if found
     */
    OutputConversationDto getConversation(UUID userId, UUID conversationId);

    /**
     * Get recently active conversations for a user.
     * @param userId the user ID
     * @param pageable the pageable containing requested paging and sorting information
     * @return page of conversations ordered by recent activity
     */
    Page<OutputConversationDto> getRecentConversationsForUser(UUID userId, Pageable pageable);

    /**
     * Get recent messages for conversation.
     *
     * @param userId the user ID
     * @param conversationId the conversation ID
     * @param pageable       the pageable containing requested paging and sorting information
     * @return page of conversations ordered by recent activity
     */
    Page<OutputChatMessageDto> getConversationMessages(UUID userId, UUID conversationId, Pageable pageable);

    /**
     * Get all participants for a conversation.
     * @param conversationId the conversation ID
     * @return list of participants
     */
    List<OutputConversationParticipantDto> getConversationParticipants(UUID conversationId);

    /**
     * Check if a user is a participant in the conversation.
     * @param conversationId the conversation ID
     * @param userId the user ID
     * @return true if the user is an active participant
     */
    boolean isUserParticipant(UUID conversationId, UUID userId);

    /**
     * Mute or unmute a conversation for a user.
     * @param conversationId the conversation ID
     * @param userId the user ID
     * @param muted true to mute, false to unmute
     */
    void muteConversation(UUID conversationId, UUID userId, boolean muted);

    /**
     * Mark all messages in the conversation as read for a user.
     *
     * @param userId the user ID
     * @param conversationId the conversation ID
     */
    void markConversationAsRead(@NonNull UUID userId, @NonNull UUID conversationId);

    /**
     * Mark all messages in the conversation as read for a user upto the message id provided.
     *
     * @param userId the user ID
     * @param conversationId the conversation ID
     * @param messageId the message ID
     */
    void markConversationUptoMessageAsRead(@NonNull UUID userId,
                                           @NonNull UUID conversationId,
                                           @NonNull UUID messageId);

    /**
     * Mark all messages in the conversation as read for a user for the bulk messages request.
     *
     * @param userId         the user ID
     * @param conversationId the conversation ID
     * @param request        the input bulk messages request
     */
    void markBulkMessagesAsRead(@NonNull UUID userId,
                                @NonNull UUID conversationId,
                                @NonNull InputBulkMessagesDto request);

    /**
     * Get conversations along with unread message information for a user.
     * @param userId the user ID
     * @return list of participant records including unread state
     */
    List<ConversationParticipant> getConversationsWithUnreadMessages(UUID userId);

    /**
     * Leave the conversation as a user.
     * @param conversationId the conversation ID
     * @param userId the user ID
     */
    void leaveConversation(UUID conversationId, UUID userId);

    /**
     * Search conversations by a free-text term (e.g., by name/description for non-direct chats).
     *
     * @param userId     the user ID
     * @param searchTerm free-text term
     * @param pageable   the pageable containing requested paging and sorting information
     * @return list of matching conversations/messages
     */
    OutputSearchResultDto searchConversations(UUID userId, String searchTerm, Pageable pageable);

    /**
     * Get aggregated counts for direct and group conversations.
     * @return conversation statistics DTO
     */
    OutputConversationStatsDto getConversationStats();

}