package com.finconsgroup.itserr.marketplace.usercommunication.dm.security;

import com.finconsgroup.itserr.messaging.security.OAuth2ClientCredentialsProvider;
import com.rabbitmq.client.impl.OAuth2ClientCredentialsGrantCredentialsProvider;
import jakarta.annotation.PostConstruct;
import lombok.extern.slf4j.Slf4j;
import org.springframework.messaging.simp.stomp.StompBrokerRelayMessageHandler;

import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;

/**
 * An Stomp OAuth2 credentials provider for RabbitMQ.
 * <p>
 * This provider authenticates with RabbitMQ using OAuth2 by exchanging client credentials
 * (client ID and secret) for an access token at the specified token endpoint. The obtained
 * token is then used for authentication with the RabbitMQ server acting as STOMP External Broker.
 * </p>
 *
 * @see OAuth2ClientCredentialsGrantCredentialsProvider
 */
@Slf4j
public class StompOAuth2ClientCredentialsWithRefreshProvider extends OAuth2ClientCredentialsProvider {

    private final StompBrokerRelayMessageHandler stompBrokerRelayMessageHandler;


    /**
     * Constructs a new OAuth2 client credentials provider with the specified OAuth2 configuration.
     *
     * @param tokenEndpointUri  the OAuth2 token endpoint URI
     * @param clientId          the OAuth2 client ID
     * @param clientSecret      the OAuth2 client secret
     * @param grantType         the OAuth2 grant type (default "client_credentials")
     * @param refreshDelay      represents how soon before expiration the token should be refreshed.
     * @param scheduledExecutor the scheduled executor to use to configure token refresh
     */
    public StompOAuth2ClientCredentialsWithRefreshProvider(String tokenEndpointUri, String clientId, String clientSecret,
                                                           String grantType, Map<String, String> parameters, Duration refreshDelay,
                                                           Duration refreshPeriod, ScheduledExecutorService scheduledExecutor,
                                                           StompBrokerRelayMessageHandler stompBrokerRelayMessageHandler) {
        super(tokenEndpointUri, clientId, clientSecret, grantType, parameters, refreshDelay, refreshPeriod, scheduledExecutor);
        this.stompBrokerRelayMessageHandler = stompBrokerRelayMessageHandler;
    }

    /**
     * Start the refresh for OAuth2 credentials
     */
    @PostConstruct
    public void start() {
        this.refresh();
    }

    @Override
    public void refresh() {
        super.refresh();

        String password = getPassword();
        if (stompBrokerRelayMessageHandler != null) {
            if (password != null && !password.isEmpty()) {
                stompBrokerRelayMessageHandler.setClientPasscode(password);
                stompBrokerRelayMessageHandler.setSystemPasscode(password);

                // need to restart the relay handler if already running,
                // so that passcode is refreshed
                if (stompBrokerRelayMessageHandler.isRunning()) {
                    stompBrokerRelayMessageHandler.stop();

                    // reset the tcp client on message handler
                    // so that a new one would be created on next start
                    stompBrokerRelayMessageHandler.setTcpClient(null);

                    // only try to start is not already running
                    if (!stompBrokerRelayMessageHandler.isRunning()) {
                        stompBrokerRelayMessageHandler.start();
                    }
                }

            } else {
                log.warn("Stomp broker relay message handler password is empty");
            }
        }
    }
}
