package com.finconsgroup.itserr.marketplace.usercommunication.dm.repository;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.OfflineMessage;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

@Repository
public interface OfflineMessageRepository extends JpaRepository<OfflineMessage, UUID> {

    /**
     * Find all undelivered messages for a specific user
     */
    List<OfflineMessage> findByRecipientUserIdAndDeliveredFalseOrderByCreatedAtAsc(UUID recipientUserId);

    /**
     * Find the earliest undelivered messages for a specific user
     */
    Optional<OfflineMessage> findTop1ByRecipientUserIdAndDeliveredFalseOrderByCreatedAtAsc(UUID recipientUserId);

    /**
     * Mark messages as delivered for a specific user
     */
    @Modifying
    @Query("UPDATE OfflineMessage om SET om.delivered = true, om.deliveredAt = :deliveredAt " +
           "WHERE om.recipientUserId = :recipientUserId AND om.delivered = false")
    int markMessagesAsDelivered(@Param("recipientUserId") UUID recipientUserId,
                                @Param("deliveredAt") Instant deliveredAt);

    /**
     * Mark specific messages as delivered
     */
    @Modifying
    @Query("UPDATE OfflineMessage om SET om.delivered = true, om.deliveredAt = :deliveredAt " +
           "WHERE om.id IN :messageIds")
    int markSpecificMessagesAsDelivered(@Param("messageIds") List<Long> messageIds,
                                        @Param("deliveredAt") Instant deliveredAt);

    /**
     * Count undelivered messages for a user
     */
    long countByRecipientUserIdAndDeliveredFalse(UUID recipientUserId);

    /**
     * Count undelivered messages for a user in a specific conversation
     */
    long countByRecipientUserIdAndConversationIdAndDeliveredFalse(UUID recipientUserId, UUID conversationId);

    /**
     * Delete delivered messages older than specified date
     */
    @Modifying
    @Query("DELETE FROM OfflineMessage om WHERE om.delivered = true AND om.deliveredAt < :cutoffDate")
    int deleteDeliveredMessagesOlderThan(@Param("cutoffDate") Instant cutoffDate);

    /**
     * Find messages by message ID and recipient (for duplicate prevention)
     */
    boolean existsByMessageIdAndRecipientUserId(UUID messageId, UUID recipientUserId);

    /**
     * Delete delivered messages for a specific user in a specific conversation
     * This is used to clean up offline messages when a user marks a conversation as read
     */
    @Modifying
    @Query("DELETE FROM OfflineMessage om WHERE om.recipientUserId = :recipientUserId " +
           "AND om.conversationId = :conversationId AND om.delivered = true")
    int deleteByRecipientUserIdAndConversationIdAndDeliveredTrue(@Param("recipientUserId") UUID recipientUserId,
                                                                 @Param("conversationId") UUID conversationId);

    /**
     * Delete undelivered messages for a specific user in a specific conversation
     * This is used to clean up offline messages when a user marks a conversation as read
     */
    @Modifying
    @Query("DELETE FROM OfflineMessage om WHERE om.recipientUserId = :recipientUserId " +
           "AND om.conversationId = :conversationId AND om.delivered = false")
    int deleteByRecipientUserIdAndConversationIdAndDeliveredFalse(@Param("recipientUserId") UUID recipientUserId,
                                                                  @Param("conversationId") UUID conversationId);

    /**
     * Delete undelivered messages for a specific user in a specific conversation upto created at.
     * This is used to clean up offline messages when a user marks a conversation as read
     */
    @Modifying
    @Query("DELETE FROM OfflineMessage om WHERE om.recipientUserId = :recipientUserId " +
            "AND om.conversationId = :conversationId AND om.delivered = false AND om.createdAt <= :uptoCreatedAt")
    int deleteByRecipientUserIdAndConversationIdAndDeliveredFalseUptoCreatedAt(
            @Param("recipientUserId") UUID recipientUserId,
            @Param("conversationId") UUID conversationId,
            @Param("uptoCreatedAt") Instant uptoCreatedAt);

    /**
     * Delete undelivered messages for a specific user in a specific conversation for message ids.
     * This is used to clean up offline messages when a user marks a conversation as read
     */
    @Modifying
    @Query("DELETE FROM OfflineMessage om WHERE om.recipientUserId = :recipientUserId " +
            "AND om.conversationId = :conversationId AND om.delivered = false AND om.messageId IN (:messageIds)")
    int deleteByRecipientUserIdAndConversationIdAndDeliveredFalseAndMessageIdIn(
            @Param("recipientUserId") UUID recipientUserId,
            @Param("conversationId") UUID conversationId,
            @Param("messageIds") Set<UUID> messageIds);
}