package com.finconsgroup.itserr.marketplace.usercommunication.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.MessageType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.EnumType;
import jakarta.persistence.Enumerated;
import jakarta.persistence.Table;
import jakarta.validation.constraints.NotBlank;
import jakarta.validation.constraints.NotNull;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.time.LocalDateTime;
import java.util.UUID;

/**
 * Entity representing a message queued for a user who was offline when it was sent.
 * Used to store and later deliver messages and notifications.
 */
@Entity
@Table(name = "offline_messages")
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(of = {"messageId", "recipientUserId"}, callSuper = true)
@NoArgsConstructor
public class OfflineMessage extends AbstractUUIDEntity {

    @Column(name = "message_id", nullable = false)
    @NotNull(message = "Message ID cannot be blank")
    private UUID messageId;

    @Column(name = "recipient_user_id", nullable = false)
    @NotNull(message = "Recipient user ID cannot be blank")
    private UUID recipientUserId;

    @Column(name = "conversation_id", nullable = false)
    @NotNull(message = "Conversation ID cannot be blank")
    private UUID conversationId;

    @Column(name = "sender_id", nullable = false)
    @NotNull(message = "Sender ID cannot be blank")
    private UUID senderId;

    @Column(name = "content", nullable = false, columnDefinition = "TEXT")
    @NotBlank(message = "Content cannot be blank")
    private String content;

    @Enumerated(EnumType.STRING)
    @Column(name = "message_type", nullable = false)
    @NotNull(message = "Message type cannot be null")
    private MessageType messageType;

    @Column(name = "created_at", nullable = false)
    private Instant createdAt;

    @Column(name = "queued_at", nullable = false)
    private LocalDateTime queuedAt;

    @Column(name = "delivered", nullable = false)
    private boolean delivered;

    @Column(name = "delivered_at")
    private LocalDateTime deliveredAt;

    public OfflineMessage(UUID messageId, UUID recipientUserId, UUID conversationId,
                          UUID senderId, String content, MessageType messageType,
                          Instant createdAt) {
        this();
        this.messageId = messageId;
        this.recipientUserId = recipientUserId;
        this.conversationId = conversationId;
        this.senderId = senderId;
        this.content = content;
        this.messageType = messageType;
        this.createdAt = createdAt;
    }

    public void setDelivered(boolean delivered) {
        this.delivered = delivered;
        if (delivered && this.deliveredAt == null) {
            this.deliveredAt = LocalDateTime.now();
        }
    }

    @Override
    public void prePersist() {
        super.prePersist();
        this.queuedAt = LocalDateTime.now();
    }
}