package com.finconsgroup.itserr.marketplace.usercommunication.dm.controller;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ValidationException;
import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.api.GroupConversationApi;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.MessageDestinations;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputCreateGroupConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputInviteUserDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputPromoteUserDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputRemoveUserDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.ChatMessage;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.Conversation;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.MessageType;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.ChatMessageProducer;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.service.GroupConversationService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.messaging.simp.SimpMessagingTemplate;
import org.springframework.web.bind.annotation.RestController;

import java.time.Instant;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.usercommunication.dm.constant.SecurityConstants.SYSTEM_USER_ID;

/**
 * REST Controller for managing group conversations
 * Provides LinkedIn-style direct messaging and group chat functionality
 */
@RestController
@RequiredArgsConstructor
@Slf4j
public class GroupConversationController implements GroupConversationApi {

    private final GroupConversationService groupConversationService;
    private final ChatMessageProducer messageProducer;
    private final SimpMessagingTemplate messagingTemplate;
    private final MessageDestinations messageDestinations;

    /**
     * Create a new group conversation
     */
    @Override
    public ResponseEntity<Conversation> createGroupConversation(InputCreateGroupConversationDto request) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        Conversation conversation = groupConversationService.createGroupConversation(
                request.getName(), request.getDescription(), userId);

        log.info("Group conversation created: {}", conversation.getId());
        return ResponseEntity.status(HttpStatus.CREATED).body(conversation);
    }

    /**
     * Invite user to group conversation
     */
    @Override
    public void inviteUserToGroup(UUID conversationId,
                                  InputInviteUserDto request) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        groupConversationService.inviteUserToGroup(conversationId, request.getUserId(), userId);
        // Send invitation notification
        ChatMessage inviteMessage = new ChatMessage();
        inviteMessage.setId(UUID.randomUUID());
        inviteMessage.setSenderId(SYSTEM_USER_ID);
        inviteMessage.setContent(request.getUserId() + " was invited to the group");
        inviteMessage.setMessageType(MessageType.SYSTEM);
        inviteMessage.setConversationId(conversationId);
        inviteMessage.setCreatedAt(Instant.now());

        messageProducer.sendChatMessage(inviteMessage);
        messagingTemplate.convertAndSend(messageDestinations.getConversationTopic(conversationId), inviteMessage);
    }

    /**
     * Remove user from group conversation
     */
    @Override
    public void removeUserFromGroup(UUID conversationId,
                                    InputRemoveUserDto request) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        boolean success = groupConversationService.removeUserFromGroup(
                conversationId, request.getUserId(), userId);

        if (success) {
            // Send removal notification
            ChatMessage removeMessage = new ChatMessage();
            removeMessage.setId(UUID.randomUUID());
            removeMessage.setSenderId(SYSTEM_USER_ID);
            removeMessage.setContent(request.getUserId() + " was removed from the group");
            removeMessage.setMessageType(MessageType.SYSTEM);
            removeMessage.setConversationId(conversationId);
            removeMessage.setCreatedAt(Instant.now());

            messageProducer.sendChatMessage(removeMessage);
            messagingTemplate.convertAndSend(messageDestinations.getConversationTopic(conversationId), removeMessage);
        } else {
            throw new WP2ValidationException("Failed to remove user from group");
        }
    }

    /**
     * Promote user to admin in group conversation
     */
    @Override
    public void promoteUserToAdmin(UUID conversationId,
                                   InputPromoteUserDto request) {
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        groupConversationService.promoteUserToAdmin(conversationId, request.getUserId(), userId);
    }

}