package com.finconsgroup.itserr.marketplace.usercommunication.dm.api;

import com.finconsgroup.itserr.marketplace.core.web.utils.OpenApiUtils;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputBulkMessagesDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.InputMuteConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputChatMessageDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationParticipantDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputConversationStatsDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputSearchResultDto;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.dto.OutputUnreadMessageSummaryDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.List;
import java.util.UUID;

/**
 * API contract for all conversation related endpoints.
 * Contains endpoint mappings and parameter annotations.
 */
@Tag(name = "Conversation", description = "Operations related to all conversations i.e. direct or group")
@SecurityRequirement(name = "BearerAuth")
@ApiResponses(value = {
        @ApiResponse(responseCode = "500", ref = OpenApiUtils.RESPONSE_INTERNAL_SERVER_ERROR)
})
public interface ConversationApi {

    @Operation(summary = "Get conversation", description = "Get conversation details by ID.")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Conversation found"),
            @ApiResponse(responseCode = "404", description = "Conversation not found")
    })
    @GetMapping(value = "/user-communication/conversations/{conversationId}",
            produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputConversationDto getConversationById(@PathVariable UUID conversationId);

    @Operation(summary = "Recent conversations for user", description = "Get user's recently active conversations.")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Conversations retrieved")
    })
    @ResponseStatus(HttpStatus.OK)
    @GetMapping(value = "/user-communication/conversations/recent/user", produces = MediaType.APPLICATION_JSON_VALUE)
    Page<OutputConversationDto> getRecentConversationsForUser(
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "lastActivity", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "DESC", required = false) Sort.Direction direction
    );

    @Operation(summary = "Get participants", description = "List participants of a conversation.",
            hidden = true)
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Participants retrieved"),
            @ApiResponse(responseCode = "404", description = "Conversation not found")
    })
    @GetMapping(value = "/user-communication/conversations/{conversationId}/participants",
            produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    List<OutputConversationParticipantDto> getConversationParticipants(@PathVariable UUID conversationId);

    @Operation(summary = "Leave conversation", description = "Leave a conversation.",
            hidden = true)
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Left conversation"),
            @ApiResponse(responseCode = "404", description = "Conversation not found"),
            @ApiResponse(responseCode = "400", description = "Invalid request")
    })
    @PostMapping("/user-communication/conversations/{conversationId}/leave")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void leaveConversation(@PathVariable UUID conversationId);

    @Operation(summary = "Mute conversation", description = "Mute notifications for a conversation.",
            hidden = true)
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Conversation muted"),
            @ApiResponse(responseCode = "404", description = "Conversation not found"),
            @ApiResponse(responseCode = "400", description = "Invalid request")
    })
    @PostMapping("/user-communication/conversations/{conversationId}/mute")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void muteConversation(@PathVariable UUID conversationId,
                          @RequestBody InputMuteConversationDto request);


    /**
     * Mark all messages as read for a user in a conversation.
     *
     * @param conversationId the conversation id
     */
    @Operation(summary = "Mark conversation as read", description = "Mark all messages as read for a user in a conversation.")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Conversation marked as read"),
            @ApiResponse(responseCode = "404", description = "Conversation not found"),
            @ApiResponse(responseCode = "400", description = "Invalid request")
    })
    @PostMapping("/user-communication/conversations/{conversationId}/read")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void markConversationAsRead(@PathVariable("conversationId") UUID conversationId);

    /**
     * Mark all messages as read for a user in a conversation upto the provided message.
     *
     * @param conversationId the conversation id
     * @param messageId the message id
     */
    @Operation(summary = "Mark conversation as read upto given message",
            description = "Mark all messages as read for a user in a conversation upto the provided message.")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Conversation marked as read"),
            @ApiResponse(responseCode = "404", description = "Conversation not found"),
            @ApiResponse(responseCode = "400", description = "Invalid request")
    })
    @PostMapping("/user-communication/conversations/{conversationId}/read-upto/{messageId}")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void markConversationUptoMessageAsRead(@PathVariable("conversationId") UUID conversationId,
                                           @PathVariable("messageId") UUID messageId);

    /**
     * Mark one or more messages as read for a user in a conversation.
     *
     * @param conversationId the conversation id
     * @param request the request containing the messages ids to mark as read
     */
    @Operation(summary = "Mark one or more messages for the conversation as read",
            description = "Mark one or more messages as read for a user in a conversation.")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Conversation marked as read"),
            @ApiResponse(responseCode = "404", description = "Conversation not found"),
            @ApiResponse(responseCode = "400", description = "Invalid request")
    })
    @PostMapping("/user-communication/conversations/{conversationId}/read-bulk")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    void markBulkMessagesAsRead(@PathVariable("conversationId") UUID conversationId,
                                @RequestBody @Valid InputBulkMessagesDto request);

    @Operation(summary = "Search conversations", description = "Search conversations by name or other criteria.")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Search results returned"),
            @ApiResponse(responseCode = "500", description = "Internal server error")
    })
    @GetMapping(value = "/user-communication/conversations/search", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputSearchResultDto searchConversations(
            @RequestParam(name = "query") String query,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "createdAt", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "DESC", required = false) Sort.Direction direction);

    @Operation(summary = "Conversation statistics", description = "Get system-wide conversation stats.",
            hidden = true)
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Statistics returned")
    })
    @GetMapping(value = "/user-communication/conversations/stats", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputConversationStatsDto getConversationStats();

    @Operation(summary = "Get conversation messages", description = "List messages for a conversation with pagination.")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Messages retrieved"),
            @ApiResponse(responseCode = "404", description = "Conversation not found")
    })
    @GetMapping(value = "/user-communication/conversations/{conversationId}/messages",
            produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    Page<OutputChatMessageDto> getConversationMessages(
            @PathVariable UUID conversationId,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "createdAt", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction);

    @Operation(summary = "Get unread summary", description = "Get summary about unread messages for a user.")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Unread summary returned")
    })
    @GetMapping(value = "/user-communication/conversations/unread/user", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputUnreadMessageSummaryDto getUnreadMessageSummary();
}
