package com.finconsgroup.itserr.marketplace.usercommunication.dm.repository;

import com.finconsgroup.itserr.marketplace.usercommunication.dm.entity.Conversation;
import com.finconsgroup.itserr.marketplace.usercommunication.dm.enums.ConversationType;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Repository;

import java.time.Instant;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import static java.util.Map.entry;

@Repository
public interface ConversationRepository extends JpaRepository<Conversation, UUID> {

    /**
     * Map of field name to column name
     */
    Map<String, String> FIELD_COLUMN_NAME_MAP = Map.ofEntries(
            entry("id", "id"),
            entry("name", "name"),
            entry("description", "description"),
            entry("createdBy", "created_by"),
            entry("createdAt", "created_at"),
            entry("lastActivity", "last_activity"),
            entry("active", "active"),
            entry("privateFlag", "is_private"),
            entry("maxParticipants", "max_participants"),
            entry("participantOneId", "participant_one_id"),
            entry("participantTwoId", "participant_two_id")
    );

    /**
     * Find a conversation by its unique conversation ID
     */
    @NonNull
    Optional<Conversation> findById(@NonNull UUID conversationId);

    /**
     * Checks if the conversation exists and is active by its unique conversation ID
     */
    boolean existsByIdAndActiveTrue(@NonNull UUID conversationId);

    /**
     * Find the requested page from all conversations for a user.
     */
    @Query("SELECT c FROM Conversation c " +
            "JOIN ConversationParticipant cp ON cp.conversationId = c.id " +
            "WHERE cp.userId = :userId AND cp.active = true AND c.active = true")
    Page<Conversation> findConversationsForUser(@Param("userId") UUID userId, Pageable pageable);

    /**
     * Find direct conversation between two users
     */
    @Query("SELECT c FROM Conversation c WHERE c.conversationType = 'DIRECT' AND c.active = true AND " +
           "((c.participantOneId = :user1Id AND c.participantTwoId = :user2Id) OR " +
           "(c.participantOneId = :user2Id AND c.participantTwoId = :user1Id))")
    Optional<Conversation> findDirectConversationBetweenUsers(@Param("user1Id") UUID user1Id, 
                                                             @Param("user2Id") UUID user2Id);

    /**
     * Search conversations by search term in the name
     */
    @Query(
            value = """
                    SELECT c.*
                    FROM conversations c
                    JOIN conversation_participants cp
                        ON    cp.conversation_id = c.id
                          AND cp.user_id = :userId AND cp.is_active = true
                    WHERE   c.active = true
                       AND (
                               (c.conversation_type = 'GROUP' AND LOWER(c.name) LIKE LOWER('%' || :term || '%'))
                          OR  (
                                  c.conversation_type = 'DIRECT'
                              AND EXISTS (
                                    SELECT 1 FROM conversation_participants cp2
                                    JOIN users u on u.id = cp2.user_id
                                    WHERE cp2.conversation_id = c.id AND cp2.is_active = true
                                    AND LOWER(u.first_name || ' ' || u.last_name) LIKE LOWER('%' || :term || '%')
                                  )
                          )
                       )
                    """,
            countQuery = """
                         SELECT COUNT(c.id)
                         FROM conversations c
                         JOIN conversation_participants cp
                             ON    cp.conversation_id = c.id
                               AND cp.user_id = :userId AND cp.is_active = true
                         WHERE   c.active = true
                            AND (
                                    (c.conversation_type = 'GROUP' AND LOWER(c.name) LIKE LOWER('%' || :term || '%'))
                               OR  (
                                       c.conversation_type = 'DIRECT'
                                   AND EXISTS (
                                         SELECT 1 FROM conversation_participants cp2
                                         JOIN users u on u.id = cp2.user_id
                                         WHERE cp2.conversation_id = c.id AND cp2.is_active = true
                                         AND LOWER(u.first_name || ' ' || u.last_name) LIKE LOWER('%' || :term || '%')
                                       )
                               )
                            )
                         """,
            nativeQuery = true
    )
    Page<Conversation> findByNameContainingIgnoreCase(@Param("userId") UUID userId, @Param("term") String term, Pageable pageable);

    /**
     * Count active conversations by type
     */
    long countByConversationTypeAndActiveTrue(ConversationType conversationType);

    /**
     * Update last activity timestamp for a conversation
     */
    @Modifying
    @Query("UPDATE Conversation c SET c.lastActivity = :timestamp WHERE c.id = :conversationId")
    void updateLastActivity(@Param("conversationId") UUID conversationId, @Param("timestamp") Instant timestamp);

    /**
     * Deactivate a conversation
     */
    @Modifying
    @Query("UPDATE Conversation c SET c.active = false WHERE c.id = :conversationId")
    void deactivateConversation(@Param("conversationId") UUID conversationId);

    /**
     * Activate a conversation
     */
    @Modifying
    @Query("UPDATE Conversation c SET c.active = true WHERE c.id = :conversationId")
    void activateConversation(@Param("conversationId") UUID conversationId);

    /**
     * Check if a conversation exists between two users (for DIRECT type)
     */
    @Query("SELECT CASE WHEN COUNT(c) > 0 THEN true ELSE false END FROM Conversation c WHERE " +
           "c.conversationType = 'DIRECT' AND c.active = true AND " +
           "((c.participantOneId = :user1Id AND c.participantTwoId = :user2Id) OR " +
           "(c.participantOneId = :user2Id AND c.participantTwoId = :user1Id))")
    boolean existsDirectConversationBetweenUsers(@Param("user1Id") UUID user1Id, @Param("user2Id") UUID user2Id);

}